/** @file libtrack/path/PathEdgeEnd.h
 *  @brief Declare the Track::PathEdgeEnd class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_PATH_PATH_EDGE_END_H
#define LIBTRACK_PATH_PATH_EDGE_END_H

#include <ostream>
#include <istream>

#include <LinearMath/btScalar.h>

#include "../Segment.h"
#include "../SegmentConnection.h"
#include "../edit_base/EdgeStrengthHandle.h"

namespace Track
{

/**Information about an Edge that is recorded for each end of the edge, but can
 * be different to multiple edges at the same vertex.
 */
class PathEdgeEnd
{
    friend std::ostream & operator<<(std::ostream & destination,
                                     const PathEdgeEnd & path_edge_end);
public:
    /** Create tied to a specific end of an edge.
     * @param edge_index The name of the edge to link to.
     * @param end The end we represent.
     */
    PathEdgeEnd(std::size_t edge_index, EditAssist::EdgeStrengthHandle::EdgeEnd end);
    
    /** Create from a stream tied to a specific end of an edge.
     * @param source The stream to extract data from. It will be advanced to
     * the end of the description of the PathEdgeEnd.
     * @param edge_index The name of the edge to link to.
     * @param end The end we represent.
     */
    PathEdgeEnd(std::istream & source,
                std::size_t edge_index,
                EditAssist::EdgeStrengthHandle::EdgeEnd end);
    
    virtual ~PathEdgeEnd();
    
    /** The strength at which to use the gradient at the vertex.
     * It is dependent on the length of the edges.
     */
    btScalar gradient_strength;
    
    /// The index of the segment connection
    std::size_t segment_connection_index;
    
    /** @todo It could make PathEdge cleaner if the relavent PathVertex was
     * linked to in here. However the graph will then store the connections of
     * each arc twice, and we'ld have to keep it up to date.
     */
    
    /// Control point to adjust gradient_strength.
    EditAssist::EdgeStrengthHandle handle;
protected:
    unsigned int file_version;
};

std::ostream & operator<<(std::ostream & destination,
                          const PathEdgeEnd & path_edge_end);

}

#endif // LIBTRACK_PATH_PATH_EDGE_END_H
