/** @file libtrack/path/PathEdgeEnd.h
 *  @brief Declare the Track::PathEdgeEnd class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_PATH_PATH_EDGE_H
#define LIBTRACK_PATH_PATH_EDGE_H

#include <ostream>
#include <istream>
#include <vector>

#include <boost/shared_ptr.hpp>

#include <LinearMath/btScalar.h>
#include <LinearMath/btTransform.h>
#include <BulletCollision/CollisionShapes/btTriangleMesh.h>

#include "../UniqueIdentifier.h"
#include "../edit_base/Selectable.h"
#include "../Segment.h"
#include "../Mesh/DrawableMesh.h"

#include "PathEdgeEnd.h"
#include "PathVertex.h"
#include "../AxisAlignedBoundingBox.h"
#include "../AABBDrawable.h"
#include "../TrackAttachment.h"
#include "../edit_base/TrackAttachmentHandle.h"

namespace Track
{

class Path;

/** Information to store along the edges.
 */
class PathEdge
    :   public UniqueIdentifier
    ,   public EditAssist::Selectable
    ,   public AABBDrawable
{
    friend std::ostream & operator<<(std::ostream & destination,
                                     const PathEdge & path_edge);
    friend AxisAlignedBoundingBox & operator|=(AxisAlignedBoundingBox & box,
                                               const PathEdge & path_edge);
public:
    PathEdge(const Theme & theme);
    PathEdge(std::istream & source, const Theme & theme);
    virtual ~PathEdge();
protected:
    const unsigned int file_version;
    const Theme & theme;
public:
    /// The index of the segment in the Theme
    std::size_t segment_index;
    /// The segment to repeat to fill space along the edge.
    const Segment * segment;
    
    /// Information about the connection to the starting vertex.
    PathEdgeEnd start;
    /// Information about the connection to the ending vertex.
    PathEdgeEnd finish;
    
    /** The draw style of the meshes produced.
     * Should be set before update is called.
     * Defaults to RM_SOLID.
     */
    DrawableMesh::RenderMode render_mode;
    
    ///@todo Addititional objects placed on track (e.g. booster, start position)
    
    /** Update various internal records.
     * You should call this after changing any related geometry: the vertices at
     * the ends of the arc, the gradient in the PathEdgeEnds, or the Segment
     * that runs along the arc.
     * After this has been called, the length, repetitions, and nav mesh, and
     * graphical meshes will be
     * up to date.
     * The objects referenced by the parameters must not be freed when this
     * object is still being used, unless this function is called again.
     * @param start_vertex_name The name of the vertex at the start of the
     * edge.
     * @param finish_vertex_name The name of the vertex at the finish end of
     * the edge.
     * @param graph The Track::Path::Graph where this edge and the vertices
     * are.
     */
    void update(std::size_t start_vertex_name,
                std::size_t finish_vertex_name,
                const Path * graph);
    
    /** Update various internal records.
     * Behaves like update(const PathVertex & source, const PathVertex & target),
     * except it uses the reuses the same source and target previously
     * passed in. The other version must have been used first to set
     * source and target.
     */
    void update();
    
    /** Get the last calculated length.
     * Use calculate_length() instead if the Edge or the vertices it connects
     * have been modified since the last time it was called.
     * If get_length is called before calculate_length, the result is undefined.
     * @return the last value calculated by calculate_length().
     */
    btScalar get_length() const;
    
    /** Get the last calculated number of repetitions of the segment needed to
     * fit the curve well.
     * Call calculate_length() to update the value returned if the segment or
     * curve's path has changed since the last call to calculate_length(). If
     * calculate_length() has never been called, the returned value is undefined.
     */
    unsigned int get_number_of_repetions() const;
    
    /** Get the transformation at a specific point along the arc. update() must
     * have been called previously to set the source and target PathVertex.
     * @param position point along the length of the arc from 0 (start) to 1
     * (finish).
     * @return transformation to apply to a vertex along the path.
     */
    btTransform get_transform(btScalar position) const;
    
    /** Get a mesh to draw part of the arc. The number of meshes is the number
     * returned by get_number_of_repetitions(). The meshes are only created when
     * update() is called.
     * @param index The 0 based index specifying which mesh to get. It is an
     * error to use a higher index than the value returned by
     * get_number_of_repetitions().
     * @todo write similar functions for the other meshes.
     */
    const boost::shared_ptr<MultiDrawableMesh> get_graphics_mesh(std::size_t index) const;
    
    virtual bool is_here(btVector3 start, btVector3 stop, btScalar radius) const;
    
    /** Get the nearest point along the curve to a line segment.
     * @param start one end of the line segment.
     * @param stop the other end of the line segment.
     * @return scalar s between 0 and 1 such that get_transform(s).origin() is
     * the closest to the line segment for any value of s between 0 and 1.
     */
    btScalar get_nearest_s(btVector3 start, btVector3 stop) const;
    
    /** Find a control point under a mouse cursor.
     * @return 0 if none near control point, otherwise an apropriate edge
     * handle.
     */
    const EditAssist::ControlPoint * get_control_point_here(btVector3 start_pos, btVector3 stop_pos, btScalar radius) const;
    
    /** Draw all ControlPoint objects adjusting this edge.
     * @see void Track::EditAssist::ControlPoint::draw() const
     */
    void draw_control_points() const;
    
    /** Get the name of the source vertex.
     * The source and target vertices must have been set with
     * update(const PathVertex & source, const PathVertex & target).
     */
    std::size_t get_start_vertex_name();
    
    /**  Get the name of the target vertex.
     * The source and target vertices must have been set with
     * update(const PathVertex & source, const PathVertex & target).
     */
    std::size_t get_finish_vertex_name();
    
    /// Add the faces you can collide with to a shape.
    void add_collision_faces(btTriangleMesh & shape) const;
    /// Add the faces you can drive on to a shape.
    void add_floor_faces(btTriangleMesh & shape) const;
    /// Add the faces the AI considers driving on to a mesh.
    void add_ai_faces(MeshFaces & mesh) const;
    
    /// Get the minimal bounding box for the graphic meshes.
    virtual AxisAlignedBoundingBox get_bounds() const;
    virtual void draw() const;
    /** Draw the parts of the edge on screen.
     * Efficently decide what to draw and draw it.
     * @param occlusion_tester The bounds of the visible area.
     */
    virtual void conditional_draw(const OcclusionTester & occlusion_tester) const;
    /// Draw the attachments on the edge.
    void draw_attachments() const;
    /// Load meshes into graphics memory.
    void make_cache() const;
    
    /** Put an attachment on the edge.
     * Precondition: Attachment removed.
     * Postcondition: Attachment added.
     */
    void insert_attachment(boost::shared_ptr<TrackAttachment> attachment);
    /** Remove an attachment from the edge.
     * Precondition: aattachment added with given name.
     * Postcondition: attachment removed.
     */
    void remove_attachment(std::size_t attachment_name);
    /** Get a list of attachments on the edge.
     */
    const std::vector<boost::shared_ptr<TrackAttachment> > & get_attachments() const;
    /** Get an attachment on the edge by name.
     * Precondition: attachment added with given name.
     */
    const TrackAttachment & get_attachment(std::size_t attachment_name) const;
    /** Modify an attachment on the edge by name.
     * Precondition: attachment added with given name.
     */
    void set_attachment(std::size_t attachment_name, const TrackAttachment & attachment);
protected:
    std::vector<boost::shared_ptr<TrackAttachment> > m_attachments;
    std::vector<boost::shared_ptr<EditAssist::TrackAttachmentHandle> > m_attachment_handles;
    btScalar length;
    unsigned int number_of_repetions;
    std::vector<boost::shared_ptr<MultiDrawableMesh> > meshes;
    const Path * path;
    std::size_t start_vertex_index;
    std::size_t finish_vertex_index;
    /// the minimal axis aligned bounding box of the edge including handles.
    AxisAlignedBoundingBox bounds;
    /// The minimal axis aligned bounding box of the edge's graphical meshes.
    AxisAlignedBoundingBox graphics_bounds;
    /// The navigation mesh
    MeshFaces m_nav_mesh;
    MeshFaces::Graph m_navigation_graph;
    
    void recreate_attachment_handles();
};

std::ostream & operator<<(std::ostream & destination,
                          const PathEdge & path_edge);

/// Expand an AxisAlignedBoundingBox minimally to contain a PathEdge with control points.
AxisAlignedBoundingBox & operator|=(AxisAlignedBoundingBox & box,
                                    const PathEdge & path_edge);

}

#endif //LIBTRACK_PATH_PATH_EDGE_H
