/** @file libtrack/path/PathEdgeEnd.h
 *  @brief Implement the Track::PathEdgeEnd class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "PathEdge.h"
#include "../stream_loader.h"
#include "../FormErrors.h"
#include "Path.h"

namespace Track
{

const unsigned int path_edge_newest_version = 2;

PathEdge::PathEdge(const Theme & theme)
    :   file_version(0)
    ,   theme(theme)
    ,   segment_index(0)
    ,   segment(0)
    ,   start(get_name(), EditAssist::EdgeStrengthHandle::EE_SOURCE)
    ,   finish(get_name(), EditAssist::EdgeStrengthHandle::EE_DESTINATION)
    ,   render_mode(DrawableMesh::RM_SOLID)
{
    
}

PathEdge::PathEdge(std::istream & source, const Theme & theme)
    :   file_version(int_from_stream(source))
    ,   theme(theme)
    ,   segment_index(theme.get_segment_index(string_from_stream(source)))
    ,   segment(&theme.get_segment(segment_index))
    ,   start(source, get_name(), EditAssist::EdgeStrengthHandle::EE_SOURCE)
    ,   finish(source, get_name(), EditAssist::EdgeStrengthHandle::EE_DESTINATION)
    ,   render_mode(DrawableMesh::RM_SOLID)
{
    if (file_version == 0)  throw CorruptedError();
    if (file_version > path_edge_newest_version) throw NewVersionError();
    if (file_version >= 2)
    {
        SaveableClassList::get_instance()->fill_vector(source, m_attachments);
        // Link all the attachments to this edge so they can be safely added and removed.
        for (std::vector<boost::shared_ptr<TrackAttachment> >::iterator it = m_attachments.begin();
              it != m_attachments.end();
              it++)
        {
            (**it).edge_name = get_name();
        }
    }
}

PathEdge::~PathEdge()
{
    
}

// calculate good sizes of steps to take in approximation of edge length.
template <int t>
struct one_over_two_to_the_power_of
{
    static const btScalar result = 0.5 * one_over_two_to_the_power_of<t - 1>::result;
};

template <>
struct one_over_two_to_the_power_of<0>
{
    static const btScalar result = 1.0;
};


void PathEdge::update(std::size_t start_vertex_index_in,
                      std::size_t finish_vertex_index_in,
                      const Path * path_in)
{
    start_vertex_index = start_vertex_index_in;
    finish_vertex_index = finish_vertex_index_in;
    path = path_in;
    update();
}

void PathEdge::update()
{
    const PathVertex & source_in = path->get_node(start_vertex_index);
    const PathVertex & target_in = path->get_node(finish_vertex_index);
    
    // reset the axis aligned bounding box we'll build it up later.
    bounds = AxisAlignedBoundingBox();
    graphics_bounds = bounds;
    
    // A cubic bezier spline isn't analytically intergratable.
    // Instead of hard numerical approximations, I'll do a piecewise linear
    // estimation.
    const std::size_t start_index = start.segment_connection_index;
    const std::size_t finish_index = finish.segment_connection_index;
    const btVector3 & p0 = source_in.get_position(start_index);
    const btVector3 & p3 = target_in.get_position(finish_index);
    
    btVector3 last_position = get_transform(0).getOrigin();
    length = 0;
    const btScalar step = one_over_two_to_the_power_of<5>::result;
    for (btScalar t = step; t < 1.0; t += step)
    {
        const btVector3 this_position = get_transform(t).getOrigin();
        length += (this_position - last_position).length();
        last_position = this_position;
    }
    
    // how many repetions of the segment do we need?
    if (segment)
    {
        // round to the nearest whole number of segments.
        number_of_repetions = int (length / segment->get_length() + 0.5);
    }
    if (number_of_repetions == 0)
    {
        // even short gaps between vertices must be filled.
        number_of_repetions = 1;
    }
    if (number_of_repetions > 64)
    {
        DEBUG_MESSAGE("Warning: large number of meshes (" << number_of_repetions << ") needed for edge " << get_name());
    }
    // make the meshes.
    meshes.clear();
    meshes.reserve(number_of_repetions);
    for (unsigned int i = 0; i < number_of_repetions; i++)
    {
        PieceDistortion transform(*this, i,
                                         segment->get_length(),
                                         segment->get_minimum_y());
        meshes.push_back(boost::shared_ptr<MultiDrawableMesh>(
            new MultiDrawableMesh(segment->get_graphics_mesh().get_distorted_faces(transform))
        ));
        meshes.back()->set_render_mode(render_mode);
        graphics_bounds |= meshes.back()->get_bounds();
    }
    
    // update control point handles.
    start.handle.update(p0, source_in.get_angle(start_index), start.gradient_strength);
    finish.handle.update(p3, target_in.get_angle(finish_index), finish.gradient_strength);
    recreate_attachment_handles();
    
    bounds |= graphics_bounds;
    bounds |= start.handle.get_position();
    bounds |= finish.handle.get_position();
    
    // Navigation mesh
    m_nav_mesh = MeshFaces();
    for (unsigned int i = 0; i < number_of_repetions; i++)
    {
        PieceDistortion transform(*this, i,
                                         segment->get_length(),
                                         segment->get_minimum_y());
        m_nav_mesh |= segment->get_ai_mesh().get_distorted_faces(transform);
    }
    m_nav_mesh.merge_doubles();
    m_nav_mesh.set_source(true, get_name());
    m_navigation_graph = m_nav_mesh.get_connectivity();
}

btScalar PathEdge::get_length() const
{
    return length;
}

unsigned int PathEdge::get_number_of_repetions() const
{
    return number_of_repetions;
}

btTransform PathEdge::get_transform(btScalar position) const
{
    const std::size_t start_index = start.segment_connection_index;
    const std::size_t finish_index = finish.segment_connection_index;
    const btScalar rposition = 1.0 - position;
    const btScalar position_squared = position * position;
    const btScalar rposition_squared = rposition * rposition;
    const btScalar position_cubed = position_squared * position;
    const btScalar rposition_cubed = rposition_squared * rposition;
    const PathVertex & start_vertex = path->get_node(start_vertex_index);
    const btVector3 & p0 = start_vertex.get_position(start_index);
    const btVector3 p1 = p0 + start_vertex.get_gradient(start_index) * start.gradient_strength;
    const PathVertex & finish_vertex = path->get_node(finish_vertex_index);
    const btVector3 & p3 = finish_vertex.get_position(finish_index);
    const btVector3 p2 = p3 + finish_vertex.get_gradient(finish_index) * finish.gradient_strength;
    const btVector3 translation =   rposition_cubed * p0
                                  + 3.0 * rposition_squared * position * p1
                                  + 3.0 * rposition * position_squared * p2
                                  + position_cubed * p3;
    // find tangent for rotation.
    const btVector3 start_tangent = p1 - p0;
    const btVector3 end_tangent = p3 - p2;
    /* The tangent's control points should be tripled, but this cancels since we
     * immediately normalise.
     */
    const btVector3 tangent = ( rposition_squared * start_tangent
                                + 2.0 * rposition * position * (p2 - p1)
                                + position_squared * end_tangent)
                               .normalize();
    // roatation is such that the xz plane is perpendicular to the tangent.
    // rotation around the tangent is specified by an up vector.
    // We get the up vector from slerp of the tangenets at the end vertices.
    const btVector3 up = start_vertex.get_up(start_index).lerp(finish_vertex.get_up(finish_index), position)
                                         .normalize();
    /** @todo improve up direction. */
    // posible better up vector, but is sometimes still unintuitive.
    /*const btVector3 up = btTransform(source->get_angle().slerp(target->get_angle(), position))
                                    (btVector3(0.0, 0.0, 1.0));*/
    // Find the cross product of tangent and up to make the other axis.
    const btVector3 cross_axis = tangent.cross(up).normalize();
    // up might not be orthoganal to the other directions, improve it:
    const btVector3 better_up = cross_axis.cross(tangent);
    btMatrix3x3 rotation(cross_axis.x(), tangent.x(), better_up.x(),
                         cross_axis.y(), tangent.y(), better_up.y(),
                         cross_axis.z(), tangent.z(), better_up.z());
    
    return btTransform(rotation, translation);
}

const boost::shared_ptr<MultiDrawableMesh> PathEdge::get_graphics_mesh(std::size_t index) const
{
    assert(index < number_of_repetions);
    return meshes[index];
}

bool PathEdge::is_here(btVector3 start, btVector3 stop, btScalar radius) const
{
    /* We should find the smallest distance between the line segment between
     * start and stop and the cubic bezier spline of the edge.
     * Instead, we'll find an approximation by using a piecewise linear
     * approximation of the cubic bezier.
     */
    btVector3 length = stop - start;
    btVector3 e2, e1;
    ///@todo we only need the position, not the full transformation.
    e2 = get_transform(0).getOrigin();
    const btScalar step = one_over_two_to_the_power_of<4>::result;
    for (btScalar t = step; t < 1.0; t += step)
    {
        e1 = e2;
        e2 = get_transform(t).getOrigin();
        // now e1->e2 is the line segment we will test.
        btVector3 ed = e2 - e1;
        btVector3 w = start - e1;
        btScalar a = length.dot(length);
        btScalar b = length.dot(ed);
        btScalar c = ed.dot(ed);
        btScalar d = length.dot(w);
        btScalar e = ed.dot(w);
        btScalar denominator = a * c - b * b;
        btScalar screen = b * e - c * d;
        btScalar section = a * e - b * d;
        // Now check if this is along the segments.
        if (screen < 0) screen = 0;
        if (screen > denominator) screen = denominator;
        if (section < 0) section = 0;
        if (section > denominator) section = denominator;
        // find distance
        btScalar distance = (w + (screen * length - section * ed) / denominator).length2();
        if (distance < radius * radius)
        {
            return true;
        }
    }
    return false;
}

float PathEdge::get_nearest_s(btVector3 start, btVector3 stop) const
{
    /* We should find the smallest distance between the line segment between
     * start and stop and the cubic bezier spline of the edge.
     * Instead, we'll find an approximation by using a piecewise linear
     * approximation of the cubic bezier.
     */
    /** @todo should share more code with is_here.
     * Calculate positions in update?
     */
    btVector3 length = stop - start;
    btVector3 e2, e1;
    ///@todo we only need the position, not the full transformation.
    e2 = get_transform(0).getOrigin();
    const btScalar step = one_over_two_to_the_power_of<4>::result;
    btScalar min_distance = std::numeric_limits<btScalar>::max();
    btScalar best_result;
    for (btScalar t = step; t < 1.0; t += step)
    {
        e1 = e2;
        e2 = get_transform(t).getOrigin();
        // now e1->e2 is the line segment we will test.
        btVector3 ed = e2 - e1;
        btVector3 w = start - e1;
        btScalar a = length.dot(length);
        btScalar b = length.dot(ed);
        btScalar c = ed.dot(ed);
        btScalar d = length.dot(w);
        btScalar e = ed.dot(w);
        btScalar denominator = a * c - b * b;
        btScalar screen = b * e - c * d;
        btScalar section = a * e - b * d;
        // Now check if this is along the segments.
        if (screen < 0) screen = 0;
        if (screen > denominator) screen = denominator;
        if (section < 0) section = 0;
        if (section > denominator) section = denominator;
        // find distance
        btScalar distance = (w + (screen * length - section * ed) / denominator).length2();
        if (distance < min_distance)
        {
            min_distance = distance;
            best_result = t + (section / denominator - 1) * step; 
        }
    }
    if (best_result < 0.0) return 0;
    else if (best_result > 1.0) return 1.0;
    else return best_result;
}

const EditAssist::ControlPoint * PathEdge::get_control_point_here(btVector3 start_pos, btVector3 stop_pos, btScalar radius) const
{
    if (start.handle.is_here(start_pos, stop_pos, radius))
    {
        return &(start.handle);
    }
    else if (finish.handle.is_here(start_pos, stop_pos, radius))
    {
        return &(finish.handle);
    }
    else
    {
        for (std::vector<boost::shared_ptr<EditAssist::TrackAttachmentHandle> >::const_iterator
                    it = m_attachment_handles.begin();
             it != m_attachment_handles.end();
             it++)
        {
            if ((**it).is_here(start_pos, stop_pos, radius))
            {
                return &(**(it));
            }
        }
    }
    // can't find a near enough control point.
    return 0;
}

void PathEdge::draw_control_points() const
{
    start.handle.draw();
    finish.handle.draw();
}

std::size_t PathEdge::get_start_vertex_name()
{
    return start_vertex_index;
}

std::size_t PathEdge::get_finish_vertex_name()
{
    return finish_vertex_index;
}

void PathEdge::add_collision_faces(btTriangleMesh & shape) const
{
    for (unsigned int i = 0; i < number_of_repetions; i++)
    {
        PieceDistortion transform(*this, i,
                                         segment->get_length(),
                                         segment->get_minimum_y());
        segment->get_floor_mesh().get_distorted_faces(transform).add_faces(shape);
        segment->get_wall_mesh().get_distorted_faces(transform).add_faces(shape);
    }
}

void PathEdge::add_floor_faces(btTriangleMesh & shape) const
{
    for (unsigned int i = 0; i < number_of_repetions; i++)
    {
        PieceDistortion transform(*this, i,
                                         segment->get_length(),
                                         segment->get_minimum_y());
        segment->get_floor_mesh().get_distorted_faces(transform).add_faces(shape);
    }
}

void PathEdge::add_ai_faces(MeshFaces & mesh) const
{
    mesh |= m_nav_mesh;    
}

AxisAlignedBoundingBox PathEdge::get_bounds() const
{
    return graphics_bounds;
}

void PathEdge::draw() const
{
    // draw everything
    assert(segment);
    for (unsigned int i = 0; i < number_of_repetions; i++)
    {
        meshes[i]->draw();
    }
    draw_attachments();
}

void PathEdge::conditional_draw(const OcclusionTester & occlusion_tester) const
{
    OcclusionTester::ViewState vs(occlusion_tester(graphics_bounds));
    switch (vs)
    {
        case OcclusionTester::VS_IN:
        case OcclusionTester::VS_PARTIAL:
            // At least partially visible.
            // We use conditional_draw on each mesh individually even if
            // completely visible, because MultiDrawableMesh selects
            // meshes based on the distance in the occlusion_tester.
            assert(segment);
            for (unsigned int i = 0; i < number_of_repetions; i++)
            {
                meshes[i]->conditional_draw(occlusion_tester);
            }
            /** @todo
             * Make Attachment an AABBDrawable, and include attachment in
             * graphics_bounds. Then attachments will not be unnecessarily drawn.
             * Requires monitoring changes in attachments.
             */
            draw_attachments();
            break;
        case OcclusionTester::VS_OUT:
            // don't draw anything, off screen.
            break;
    }
}

void PathEdge::draw_attachments() const
{
    // Draw attachments.
    if (render_mode == DrawableMesh::RM_WIREFRAME)
    {
        glEnable(GL_TEXTURE_2D);
    }
    for (std::vector<boost::shared_ptr<TrackAttachment> >::const_iterator it = m_attachments.begin();
         it != m_attachments.end();
         it++)
    {
        glPushMatrix();
            btScalar mat[16];
            mat[15] = 1.0;
            (**it).get_global_transform().getOpenGLMatrix(mat);
            glMultMatrixf(mat);
            (**it).draw();
        glPopMatrix();
    }
    if (render_mode == DrawableMesh::RM_WIREFRAME)
    {
        glDisable(GL_TEXTURE_2D);
    }
}

void PathEdge::make_cache() const
{
    for (unsigned int i = 0; i < number_of_repetions; i++)
    {
        meshes[i]->make_cache();
    }
}

void PathEdge::insert_attachment(boost::shared_ptr<TrackAttachment> attachment)
{
    m_attachments.push_back(attachment);
    recreate_attachment_handles();
}

void PathEdge::remove_attachment(std::size_t attachment_name)
{
    std::vector<boost::shared_ptr<TrackAttachment> >::iterator attach_it;
    for (attach_it = m_attachments.begin();
         (**attach_it).get_name() != attachment_name;
         attach_it++)
    {
        assert(attach_it != m_attachments.end());
    }
    assert(attach_it != m_attachments.end());
    m_attachments.erase(attach_it);
    // the attachment handles need to know their new index. Recreate them.
    recreate_attachment_handles();
}

const TrackAttachment & PathEdge::get_attachment(std::size_t attachment_name) const
{
    std::vector<boost::shared_ptr<TrackAttachment> >::const_iterator attach_it;
    for (attach_it = m_attachments.begin();
         (**attach_it).get_name() != attachment_name;
         attach_it++)
    {
        assert(attach_it != m_attachments.end());
    }
    assert(attach_it != m_attachments.end());
    return **attach_it;
}

void PathEdge::set_attachment(std::size_t attachment_name,
                              const TrackAttachment & attachment)
{
    std::vector<boost::shared_ptr<TrackAttachment> >::iterator attach_it;
    unsigned int index = 0;
    for (attach_it = m_attachments.begin();
         (**attach_it).get_name() != attachment_name;
         attach_it++, index++)
    {
        assert(attach_it != m_attachments.end());
    }
    assert(attach_it != m_attachments.end());
    (**attach_it) = attachment;
    // Set correct handle position.
    btTransform transform = get_transform(attachment.get_t_position());
    btVector3 pos = transform(btVector3(attachment.get_lateral_position(),
                              0.0,
                              attachment.get_vertical_position()));
    m_attachment_handles[index]->set_position(pos);
    // set attachment's global to local transformation.
    transform.setOrigin(pos);
    (**attach_it).set_global_transform(transform);
    
}

void PathEdge::recreate_attachment_handles()
{
    m_attachment_handles.clear();
    for (std::size_t i = 0; i < m_attachments.size(); i++)
    {
        m_attachment_handles.push_back(
            boost::shared_ptr<EditAssist::TrackAttachmentHandle>(
                new EditAssist::TrackAttachmentHandle(*this, i)));
        EditAssist::TrackAttachmentHandle & handle = *(m_attachment_handles.back());
        const TrackAttachment & attachment = *(m_attachments[i]);
        
        btTransform transform = get_transform(attachment.get_t_position());
        btVector3 pos = transform(btVector3(attachment.get_lateral_position(),
                                            0.0,
                                            attachment.get_vertical_position()));
        handle.set_position(pos);
        // also tell the attachment its global transform, since if the handle
        // is off the transform will be too.
        transform.setOrigin(pos);
        m_attachments[i]->set_global_transform(transform);
    }
}

const std::vector<boost::shared_ptr<TrackAttachment> > & PathEdge::get_attachments() const
{
    return m_attachments;
}

std::ostream & operator<<(std::ostream & destination, const PathEdge & path_edge)
{
    destination << path_edge_newest_version << ' ';
    string_to_stream(destination, path_edge.segment->get_name());
    destination << ' ' << path_edge.start
                << ' ' << path_edge.finish
                << ' ';
    SaveableClassList::get_instance()->write_vector(destination, path_edge.get_attachments());
    return destination;
}

AxisAlignedBoundingBox & operator|=(AxisAlignedBoundingBox & box,
                                    const PathEdge & path_edge)
{
    box |= path_edge.bounds;
    return box;
}

}
