/** @file libtrack/path/Path.h
 *  @brief Declare the Track::Path class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_PATH_PATH_H_
#define LIBTRACK_PATH_PATH_H_

#include <istream>
#include <ostream>

/* gcc >= 4.3 Depreciates hash_set & hash_map, causing warnings to be emited
 * when including boost graph library headers.
 * We don't need the hash-based storage selectors anyway, so turn them off.
 */
#define BOOST_NO_HASH
#include <boost/graph/adjacency_list.hpp>
#undef BOOST_NO_HASH

#include <LinearMath/btScalar.h>
#include <BulletCollision/CollisionShapes/btTriangleMesh.h>

#include "../Theme.h"
#include "../AxisAlignedBoundingBox.h"
#include "../Drawable.h"

#include "PathEdge.h"
#include "PathVertex.h"

namespace Track
{

class Track;

/** A graph describing the track layout.
 */
class Path
    :   public Drawable
    ,   public AABBBounded
{
    friend std::ostream & operator<<(std::ostream & destination,
                                     const Path & path);
public:
    /// initalise from a stream
    Path(std::istream & source, const Theme & theme, const Track * track, bool editor = false);
    /// Make an empty path.
    Path(const Theme & theme, const Track * track, bool editor = false);
    virtual ~Path();
    
    /// Graph of track layout.
    typedef boost::adjacency_list <boost::vecS, boost::vecS, 
                                   boost::bidirectionalS,
                                   PathVertex, PathEdge> Graph;
    Graph graph;
    
    /** Get the axis aligned bounding box of the vertices and control points in
     * the path with some padding.
     * 
     * @todo return the axis aligned bounding box of the path instead.
     * @todo Monitor changes to graph, then only calculate when it has changed.
     */
    AxisAlignedBoundingBox get_bounding_box() const;
    
    /// find a graph node by ID
    PathVertex & get_node(const std::size_t index);
    /// find a graph node by ID
    const PathVertex & get_node(const std::size_t index) const;
    
    /// find a graph node descriptor by ID
    Path::Graph::vertex_descriptor get_node_descriptor(const std::size_t index);
    /// find a graph node descriptor by ID
    const Path::Graph::vertex_descriptor get_node_descriptor(const std::size_t index) const;
    
    
    /// find a graph edge by ID
    PathEdge & get_edge(const std::size_t index);
    /// find a graph edge by ID
    const PathEdge & get_edge(const std::size_t index) const;
    
    /// find a graph edge descriptor by ID
    Path::Graph::edge_descriptor get_edge_descriptor(const std::size_t index);
    /// find a graph edge descriptor by ID
    const Path::Graph::edge_descriptor get_edge_descriptor(const std::size_t index) const;
    
    /** Set the lengths of the rotation handles present in the path's
     * vertices.
     * @param handle_length The length the handles extend from the point they
     * rotate.
     */
    void set_handle_lengths(btScalar handle_length);
    
    /** Update the edges connected to a particular vertex.
     * This recalculates the length, number of segments, and meshes. You
     * should call this after changing a vertex's position, segment, or angle.
     * @param vertex_descriptor The vertex descriptor of the vertex that was
     * changed.
     */
    void update_connected_edges(Path::Graph::vertex_descriptor vertex_descriptor);
    
    /** Find the nearest SegmentConnectionHandle to the given position.
     */
    const EditAssist::SegmentConnectionHandle *get_nearest_segment_connection_handle(btVector3 position) const;
    
    /** Find the nearest SegmentConnectionHandle to the given line.
     * @param position A point on the line
     * @param normal A normal vector pointing down the line.
     */
    const EditAssist::SegmentConnectionHandle *get_nearest_segment_connection_handle(btVector3 position, btVector3 normal) const;
    
    /// True to make paths appear wireframe, false for solid shading.
    bool editor;
    
    /// Add the surfaces you can collide with to a shape. 
    void add_collision_faces(btTriangleMesh & shape) const;
    /// Add the surfaces you should be able to drive on to a shape.
    void add_floor_faces(btTriangleMesh & shape) const;
    /// Add the surface AI considers to a mesh.
    void add_ai_faces(MeshFaces & mesh) const;
    
    /** return the name of the edge which cars should start the race on.
     * If no edge has been provided, returns 0.
     * The first edge becomes the starting edge when loading from a stream,
     * so if you get a track this way you don't need to check for 0.
     */
    unsigned long int get_starting_edge() const;
    
    /// change the edge cars start on.
    void set_starting_edge(unsigned long int new_edge_name);
    
    /** Find the plane where start/finish line is on.
     * Finds the plane of the start / finish line in Hessian normal form.
     * @param normal Variable to be set to the unit normal of the plane
     * @param distance Variable to be set to the distance of the plane from the origin.
     */
    void find_start_plane(btVector3 & normal, btScalar & distance) const;
    /// Get the centre of the disc used as the start finish line.
    btVector3 find_start_position() const;
    
    // for Track::Drawable.
    virtual void draw() const;
    virtual void conditional_draw(const OcclusionTester & occlusion_tester) const;
    // for AABBBounded.
    virtual AxisAlignedBoundingBox get_bounds() const;
private:
    const unsigned int file_version;
    const Theme & theme;
    Path::Graph::vertex_descriptor get_node_descriptor_internal(const std::size_t index) const;
    Path::Graph::edge_descriptor get_edge_descriptor_internal(const std::size_t index) const;
    /// The name of the edge to start the cars on, or 0 if not set properly.
    unsigned long int starting_edge;
};

std::ostream & operator<<(std::ostream & destination,
                          const Path & path);

}

#endif // LIBTRACK_PATH_PATH_H_
