/** @file libtrack/path/Path.cpp
 *  @brief Implement the Track::Path class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "Path.h"
#include "../stream_loader.h"
#include "../FormErrors.h"

#include <limits>

namespace Track
{

const unsigned int path_newest_version = 2;

Path::Path(std::istream & source, const Theme & theme, const Track * track,
           bool editor)
    :   file_version(int_from_stream(source))
    ,   editor(editor)
    ,   theme(theme)
    ,   starting_edge(0)
{
    if (file_version == 0) throw CorruptedError();
    if (file_version > path_newest_version) throw NewVersionError();
    // We store the graph in the stream as a vector of vertices, followed by a
    // vector of edges.
    std::size_t number_of_vertices;
    source >> number_of_vertices;
    // add vertices with descriptors
    for (std::size_t vertex_index = 0;
         vertex_index < number_of_vertices;
         vertex_index++)
    {
        boost::add_vertex(PathVertex(source, theme, track), graph);
    }
    
    // now on to the edges
    std::size_t number_of_edges;
    source >> number_of_edges;
    for (std::size_t edge_index = 0; edge_index < number_of_edges; edge_index++)
    {
        // Read vertex indices
        std::size_t vert_index_start, vert_index_end;
        source >> vert_index_start >> vert_index_end;
        if (   vert_index_start > num_vertices(graph)
            || vert_index_end   > num_vertices(graph))
        {
            // File refers to vertices that don't exist, so it must be invalid.
            throw CorruptedError();
        }
        
        // Read edge data
        PathEdge edge_data(source, theme);
        
        std::pair<Graph::edge_descriptor, bool> new_edge = 
        add_edge(vert_index_start, vert_index_end, edge_data, graph);
        // update the edge's information (length, meshes, and so on).
        PathEdge & new_edge_data = graph[new_edge.first];
        new_edge_data.render_mode = editor ? DrawableMesh::RM_WIREFRAME : DrawableMesh::RM_SOLID;
        new_edge_data.update(graph[vert_index_start].get_name(),
                             graph[vert_index_end].get_name(),
                             this);
        if (file_version >= 2)
        {
            bool is_start_edge;
            source >> is_start_edge;
            if(is_start_edge) starting_edge = new_edge_data.get_name();
        }
    }
    
    if (starting_edge == 0)
    {
        // Starting edge was not recorded.
        // Set it to the first edge if there is one.
        if (boost::num_edges(graph) > 0)
        {
            starting_edge = graph[*(boost::edges(graph).first)].get_name();
        }
    }
}
    
Path::Path(const Theme & theme, const Track * track, bool editor)
    :   file_version(0)
    ,   editor(editor)
    ,   theme(theme)
    ,   starting_edge(0)
{
}

Path::~Path()
{
    
}

std::ostream & operator<<(std::ostream & destination, const Path & path)
{
    destination << path_newest_version << ' ';
    // vertices
    destination << boost::num_vertices(path.graph) << ' ';
    typedef boost::graph_traits<Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range  = boost::vertices(path.graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        destination << path.graph[*(vertex_range.first)] << ' ';
    }
    
    destination << boost::num_edges(path.graph) << ' ';
    typedef boost::graph_traits<Path::Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range  = boost::edges(path.graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        // output the vertex descriptor of the source and target vertices,
        // so the graph can be reconstructed.
        destination << boost::source(*(edge_range.first), path.graph) << ' '
                    << boost::target(*(edge_range.first), path.graph) << ' '
        // Now output the the PathEdge information.
                    << path.graph[*(edge_range.first)] << ' '
        // is this the starting edge?
        // we write it like this so we can safely load a Path where the 
        // starting edge has been deleted, even though it saves space just to
        // identify the right one at the end.
                    << (path.graph[*(edge_range.first)].get_name() == path.starting_edge) << ' ';
    }
    return destination;
}

AxisAlignedBoundingBox Path::get_bounding_box() const
{
    AxisAlignedBoundingBox result;
    
    // find the bounding box of all the vertices.
    typedef boost::graph_traits<Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range  = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        result |= graph[*(vertex_range.first)];
    }
    
    // now expand to include control points
    typedef boost::graph_traits<Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range  = boost::edges(graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        const Graph::edge_descriptor edge_descriptor = *(edge_range.first);
        const PathEdge & edge = graph[edge_descriptor];
        result |= edge;
    }
    
    // add some padding.
    result.add_border(20.0);
    
    return result;
}

PathVertex & Path::get_node(const std::size_t index)
{
    return graph[get_node_descriptor(index)];
}

const PathVertex & Path::get_node(const std::size_t index) const
{
    return graph[get_node_descriptor(index)];
}

Path::Graph::vertex_descriptor Path::get_node_descriptor(const std::size_t index)
{
    return get_node_descriptor_internal(index);
}

const Path::Graph::vertex_descriptor Path::get_node_descriptor(const std::size_t index) const
{
    return get_node_descriptor_internal(index);
}

Path::Graph::vertex_descriptor Path::get_node_descriptor_internal(const std::size_t index) const
{
    /// @todo Use a more efficent data structure?
    typedef boost::graph_traits<Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range  = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        const PathVertex & vertex = graph[*(vertex_range.first)];
        if (vertex.get_name() == index)
        {
            return *(vertex_range.first);
        }
    }
    DEBUG_MESSAGE("No node has index " << index << ", aborting.");
    assert(false);
    throw;
}


PathEdge & Path::get_edge(const std::size_t index)
{
    return graph[get_edge_descriptor(index)];
}

const PathEdge & Path::get_edge(const std::size_t index) const
{
    return graph[get_edge_descriptor(index)];
}

Path::Graph::edge_descriptor Path::get_edge_descriptor(const std::size_t index)
{
    return get_edge_descriptor_internal(index);
}

const Path::Graph::edge_descriptor Path::get_edge_descriptor(const std::size_t index) const
{
    return get_edge_descriptor_internal(index);
}

Path::Graph::edge_descriptor Path::get_edge_descriptor_internal(const std::size_t index) const
{
    /// @todo Use a more efficent data structure?
    typedef boost::graph_traits<Path::Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range  = boost::edges(graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        const PathEdge & edge = graph[*(edge_range.first)];
        if (edge.get_name() == index)
        {
            return *(edge_range.first);
        }
    }
    DEBUG_MESSAGE("No edge has index " << index << ", aborting.");
    assert(false);
    throw;
}

void Path::set_handle_lengths(btScalar handle_length)
{
    typedef boost::graph_traits<Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range  = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        graph[*(vertex_range.first)].set_handle_lengths(handle_length);
    }
}

void Path::update_connected_edges(Path::Graph::vertex_descriptor vertex_descriptor)
{
    // edges out of the vertex
    typedef boost::graph_traits<Graph>::out_edge_iterator OutEdgeIterator;
    std::pair<OutEdgeIterator, OutEdgeIterator> out_edge_range;
    for (out_edge_range = boost::out_edges(vertex_descriptor, graph);
         out_edge_range.first != out_edge_range.second;
         out_edge_range.first++)
    {
        PathEdge & edge = graph[*(out_edge_range.first)];   
        edge.update();
    }
    // edges into the vertex
    typedef boost::graph_traits<Graph>::in_edge_iterator InEdgeIterator;
    std::pair<InEdgeIterator, InEdgeIterator> in_edge_range;
    for (in_edge_range = boost::in_edges(vertex_descriptor, graph);
         in_edge_range.first != in_edge_range.second;
         in_edge_range.first++)
    {
        PathEdge & edge = graph[*(in_edge_range.first)];   
        edge.update();
    }
}

const EditAssist::SegmentConnectionHandle * Path::get_nearest_segment_connection_handle(btVector3 position) const
{
    btScalar best_distance = std::numeric_limits<btScalar>::max();
    typedef boost::graph_traits<Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    btVector3 best_handle_position(0, 0, 0);
    const EditAssist::SegmentConnectionHandle * best_handle = 0;
    for (vertex_range  = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        const PathVertex & vertex = graph[*(vertex_range.first)];
        const std::vector<boost::shared_ptr<EditAssist::SegmentConnectionHandle> > & handles =
            vertex.get_connection_handles();
        
        for (std::vector<boost::shared_ptr<EditAssist::SegmentConnectionHandle> >::const_iterator it = handles.begin();
             it != handles.end();
             it++)
        {
            btScalar this_distance = (**it).get_position().distance2(position);
            if (this_distance < best_distance)
            {
                best_handle = &(**it);
                best_distance = this_distance;
            }
        }
    }
    return best_handle;
}

const EditAssist::SegmentConnectionHandle * Path::get_nearest_segment_connection_handle(btVector3 position, btVector3 normal) const
{
    /// @todo take into account normal.
    return get_nearest_segment_connection_handle(position);
}

void Path::add_collision_faces(btTriangleMesh & shape) const
{
    typedef boost::graph_traits<Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range  = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        graph[*(vertex_range.first)].add_collision_faces(shape);
    }
    
    typedef boost::graph_traits<Path::Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range  = boost::edges(graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        graph[*(edge_range.first)].add_collision_faces(shape);
    }
}

void Path::add_floor_faces(btTriangleMesh & shape) const
{
    typedef boost::graph_traits<Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range  = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        graph[*(vertex_range.first)].add_floor_faces(shape);
    }
    
    typedef boost::graph_traits<Path::Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range  = boost::edges(graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        graph[*(edge_range.first)].add_floor_faces(shape);
    }
}

void Path::add_ai_faces(MeshFaces & mesh) const
{
    typedef boost::graph_traits<Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range  = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        graph[*(vertex_range.first)].add_ai_faces(mesh);
    }
    
    typedef boost::graph_traits<Path::Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range  = boost::edges(graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        graph[*(edge_range.first)].add_ai_faces(mesh);
    }
}

unsigned long int Path::get_starting_edge() const
{
    return starting_edge;
}

void Path::set_starting_edge(unsigned long int new_edge_name)
{
    starting_edge = new_edge_name;
}


void Path::find_start_plane(btVector3 & normal, btScalar & distance) const
{
    assert(starting_edge);
    const PathEdge & edge(get_edge(starting_edge));
    btTransform transform = edge.get_transform(1.0);
    // plane goes through origin
    btVector3 origin = transform.getOrigin();
    normal = transform(btVector3(0.0, 1.0, 0.0)) - origin;
    distance = -normal.dot(origin);
}

btVector3 Path::find_start_position() const
{
    assert(starting_edge);
    const PathEdge & edge(get_edge(starting_edge));
    return edge.get_transform(1.0).getOrigin();
}

void Path::draw() const
{
    // edges
    typedef boost::graph_traits<Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range = boost::edges(graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        const PathEdge & edge = graph[*(edge_range.first)];
        edge.draw();
    }
    // vertices.
    typedef boost::graph_traits<Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        const PathVertex & vertex = graph[*(vertex_range.first)];
        vertex.draw();
    }
}

void Path::conditional_draw(const OcclusionTester & occlusion_tester) const
{
    /** @todo Some sort of tree data structure would probably be more
     * efficent for establishing viewable static geometry.
     */
    // edges
    typedef boost::graph_traits<Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range = boost::edges(graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        const PathEdge & edge = graph[*(edge_range.first)];
        edge.conditional_draw(occlusion_tester);
    }
    // vertices.
    typedef boost::graph_traits<Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        const PathVertex & vertex = graph[*(vertex_range.first)];
        vertex.conditional_draw(occlusion_tester);
    }
}

AxisAlignedBoundingBox Path::get_bounds() const
{
    // bounds of all components.
    AxisAlignedBoundingBox bounds;
        /** @todo Some sort of tree data structure would probably be more
     * efficent for establishing viewable static geometry.
     */
    // edges
    typedef boost::graph_traits<Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range = boost::edges(graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        const PathEdge & edge = graph[*(edge_range.first)];
        bounds |= edge;
    }
    // vertices.
    typedef boost::graph_traits<Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range = boost::vertices(graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        const PathVertex & vertex = graph[*(vertex_range.first)];
        bounds |= vertex;
    }
    return bounds;
}

}
