/** @file libtrack/edit_base/VertexRotationHandle.h
 *  @brief Declare the Track::EditAssist::VertexRotationHandle class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef VERTEX_ROTATION_HANDLE_H_
#define VERTEX_ROTATION_HANDLE_H_

#include <LinearMath/btQuaternion.h>

#include "RotationHandle.h"

namespace Track
{

namespace EditAssist
{

class VertexRotationHandle
    :   public RotationHandle
{
public:
    /// Directions from centre point.
    enum Direction
    {
        /// +z direction
        DIR_UP = 0,
        /// +y direction
        DIR_FORWARD = 1
    };
    /** Create for indicating a direction at a vertex.
     * @param vertex_id The name of the vertex to associate with.
     * @param direction The direction to indicate.
     */
    VertexRotationHandle(std::size_t vertex_id, Direction direction = DIR_FORWARD);
    
    /** Make a DocumentDelta performing the rotation.
     * The rotation is picked to keep the angle as close as it can be to its
     * current value while moving the VertexRotationHandle to the requested
     * position.
     * @param position The desired new location of the vertex. This should
     * have been generated by Snap().
     * @return shared pointer to a new DocumentDelta that rotates the
     * associated vertex to the most appropriate angle.
     */
    virtual boost::shared_ptr<Document::DocumentDelta> make_delta(btVector3 position) const;
    /** Update the angle of the vertex.
     * This must be called for make_delta() to work.
     * This will automatically update the position to reflect the angle,
     * centre, and length.
     * @param angle The new angle of the vertex.
     */
    void update_angle(btQuaternion angle);
    
    virtual void draw() const;
protected:
    /// The name of the associated vertex.
    std::size_t vertex_id;
    /// Which direction the handle represents.
    Direction direction;
    /// The current angle of the vertex.
    btQuaternion rotation;
};

}

}

#endif // VERTEX_ROTATION_HANDLE_H_
