/** @file libtrack/edit_base/VertexRotationHandle.cpp
 *  @brief Implement the Track::EditAssist::VertexRotationHandle class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "VertexRotationHandle.h"

#include <cmath>

#include <GL/gl.h>

#include <LinearMath/btQuaternion.h>

#include "../path/Path.h"
#include "../document/RotateVertexDelta.h"

#include <Debug.h>

namespace Track
{

namespace EditAssist
{


VertexRotationHandle::VertexRotationHandle(std::size_t vertex_id, Direction direction)
    :   vertex_id(vertex_id)
    ,   direction (direction)
{
    length = 1.0;
}

boost::shared_ptr<Document::DocumentDelta> VertexRotationHandle::make_delta(btVector3 new_position) const
{
    /* Find an angle with minimal difference to the last one that fits the new
     * position.
     */
    btVector3 new_vector = (new_position - centre) / length;
    btVector3 old_vector = (position - centre) / length;
    btVector3 cross = old_vector.cross(new_vector);
    bool obtuse = (new_vector + old_vector).length2() < 2;
    btScalar sin_angle = cross.length();
    
    // some floating point rounding fixes:
    if (std::fabs(sin_angle) < 0.002)
    {
        if (!obtuse)
        {
            DEBUG_MESSAGE("Rotation too small, ignoring.");
            // Didn't rotate very far.
            // Vector is very short: just act as if there was no rotation to
            // avoid problems normalising it.
            return boost::shared_ptr<Document::DocumentDelta>
            (
                new Document::RotateVertexDelta
                (
                    Document::VertexAngleFinder(vertex_id),
                    rotation
                )
            );
        }
        else
        {
            // We can't rely on the direction we are rotating about, but
            // need to rotate about 180 degrees. Guess a direction.
            DEBUG_MESSAGE("Rotation too close to 180 degrees, guessing direction.");
            return boost::shared_ptr<Document::DocumentDelta>
            (
                new Document::RotateVertexDelta
                (
                    Document::VertexAngleFinder(vertex_id),
                    /// @todo In 180 degree case use another handle's direction?
                    rotation * btQuaternion(btVector3(0, 0, 1), M_PI)
                )
            );
        }
    }
    // sin_angle might be just outside the valid range.
    if (sin_angle > 1.0) sin_angle = 1.0;
    
    btScalar angle = std::asin(sin_angle);
    if (obtuse)
    {
        angle = M_PI - angle;
    }
    
    cross.normalize();
    btQuaternion rotation_difference(cross, angle);
    btQuaternion new_angle(rotation_difference * rotation);
    
    return boost::shared_ptr<Document::DocumentDelta>
    (
        new Document::RotateVertexDelta
        (
            Document::VertexAngleFinder(vertex_id),
            new_angle
        )
    );
}

void VertexRotationHandle::update_angle(btQuaternion angle_in)
{
    /* The position is centre position + the a vector in the direction the
     * handle indicates with the requested length.
     */
    rotation = angle_in;
    position = centre + length * 
        btTransform(angle_in)(direction == DIR_FORWARD ? btVector3(0, 1, 0) : btVector3(0, 0, 1));
}

void VertexRotationHandle::draw() const
{
    switch (direction)
    {
        case DIR_FORWARD:
            glColor4f(0.0, 1.0, 0.0, 1.0);
            break;
        case DIR_UP:
            glColor4f(1.0, 0.0, 0.0, 1.0);
            break;
    }
    RotationHandle::draw();
}

}

}

