/** @file libtrack/edit_base/TrackAttachmentHandle.cpp
 *  @brief Implement the Track::EditAssist::TrackAttachmentHandle class. 
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "TrackAttachmentHandle.h"
#include "../document/MoveTrackAttachmentDelta.h"
#include "../path/PathEdge.h"
#include "../document/InsertTrackAttachmentDelta.h"

namespace Track
{

namespace EditAssist
{

TrackAttachmentHandle::TrackAttachmentHandle(const PathEdge & edge, std::size_t attachment_index)
    :   m_edge(edge)
    ,   m_attachment_index(attachment_index)
{
}

void TrackAttachmentHandle::draw() const
{
    // Will probably be covered by the attachment in most views,
    // but is useful from some angles.
#ifndef HAVE_GLES
    glColor3f(0.0, 1.0, 1.0);
    glBegin(GL_POINTS);
        glVertex3f(position.x(), position.y(), position.z());
    glEnd();
#endif
}

void TrackAttachmentHandle::snap(btVector3 & position, btVector3 normal) const
{
    btScalar s = m_edge.get_nearest_s(position, position + normal);
    position = m_edge.get_transform(s).getOrigin();
}

boost::shared_ptr<Document::DocumentDelta> TrackAttachmentHandle::make_delta(btVector3 position) const
{
    // position should have been snapped with snap,
    // so it doesn't really matter that we do not know the direction.
    // but the line should have non-zero length for the maths to work.
    btScalar s = m_edge.get_nearest_s(position, position + btVector3(0.001, 0.001, 0.001));
    
    TrackAttachment & attachment = *(m_edge.get_attachments()[m_attachment_index]);
    btVector3 new_pos(s, attachment.get_lateral_position(),
                      attachment.get_vertical_position());
    
    return boost::shared_ptr<Document::DocumentDelta>(new
        Document::MoveTrackAttachmentDelta(
            Document::TrackAttachmentPositionFinder(m_edge.get_name(),
                                                    attachment.get_name()),
            new_pos));
}

boost::shared_ptr<Document::DocumentDelta> TrackAttachmentHandle::make_remove_delta() const
{
    TrackAttachment & attachment = *(m_edge.get_attachments()[m_attachment_index]);
    return boost::shared_ptr<Document::DocumentDelta>(new
        Document::RemoveTrackAttachmentDelta(
            Document::InsertRemoveTrackAttachmentIdentifier(
                attachment.get_name(),
                m_edge.get_name())));
}

} // namespace EditAssist

} // namespace Track
