/** @file libtrack/edit_base/SegmentConnectionHandle.cpp
 *  @brief Implement the Track::EditAssist::SegmentConnectionHandle class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "SegmentConnectionHandle.h"

#include <GL/gl.h>

#include "../document/InsertEdgeDelta.h"

#include "../Track.h"


namespace Track
{

namespace EditAssist
{

SegmentConnectionHandle::SegmentConnectionHandle(const Track & track, std::size_t vertex_name, std::size_t connection_index, const SegmentConnection * segment_connection)
    :   track(track)
    ,   vertex_name(vertex_name)
    ,   connection_index(connection_index)
    ,   segment_connection(segment_connection)
{
    DEBUG_MESSAGE("Created " << this);
}

SegmentConnectionHandle::~SegmentConnectionHandle()
{
    DEBUG_MESSAGE("Destroyed " << this);
}

void SegmentConnectionHandle::draw() const
{
#ifndef HAVE_GLES
    glColor3ub(255, 0, 255);
    glBegin(GL_POINTS);
        glVertex3f(position.x(), position.y(), position.z());
    glEnd();
#endif
}

void SegmentConnectionHandle::snap(btVector3 & new_position, btVector3 normal) const
{
    /// snap to the nearest other SegmentConnectionHandle.
    const SegmentConnectionHandle * other = track.get_path().get_nearest_segment_connection_handle(new_position, normal);
    if (other)
    {
        new_position = other->position;
    }
}

boost::shared_ptr<Document::DocumentDelta> SegmentConnectionHandle::make_delta(btVector3 new_position) const
{
    const Path & path = track.get_path();
    const SegmentConnectionHandle * other = path.get_nearest_segment_connection_handle(new_position);
    if (other && other != this)
    {
        btScalar gradient_strength = position.distance(other->position) / 2.5;
        const Theme & theme = track.get_theme();
        PathEdge edge(theme);
        edge.segment_index = theme.get_default_segment_index();
        edge.segment = &(theme.get_segment(edge.segment_index));
        edge.start.segment_connection_index = connection_index;
        edge.start.gradient_strength = gradient_strength;
        edge.finish.segment_connection_index = other->connection_index;
        edge.finish.gradient_strength = gradient_strength;
        edge.render_mode = path.editor ? DrawableMesh::RM_WIREFRAME : DrawableMesh::RM_SOLID;
        DEBUG_MESSAGE(this << " is about to update its new edge connecting v" << vertex_name << ".c" << connection_index << " to v" << other->vertex_name << ".c" << other->connection_index << ".");
        edge.update(vertex_name, other->vertex_name, &(path));
        return boost::shared_ptr<Document::DocumentDelta>(new Document::InsertEdgeDelta(edge));
    }
    // no valid connection to make. Return null pointer.
    return boost::shared_ptr<Document::DocumentDelta>();
}

} // namespace EditAssist

} // namespace Track
