/** @file libtrack/edit_base/RotationHandle.cpp
 *  @brief Implement the Track::EditAssist::RotationHandle class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include <RotationHandle.h>

#include <cmath>

#include <GL/gl.h>

namespace Track
{

namespace EditAssist
{

void RotationHandle::snap(btVector3 & proposed_position, btVector3 normal) const
{
    // first, find screen space length from centre to proposed position.
    btVector3 proposed_vector = (proposed_position - centre);
    btScalar proposed_length2 = proposed_vector.length2();
    // height above the plane going through centre with normal normal.
    btScalar plane_height = proposed_vector.dot(normal);
    // use pythagoras' theorem to find square of length across plane.
    btScalar screen_length2 = proposed_length2 - plane_height * plane_height;
    // Too long?
    if (screen_length2 >= length * length)
    {
        // make perpendicular to normal.
        btVector3 new_vector = normal.cross(proposed_vector.cross(normal));
        if (proposed_position.length2() > 0.0)
        {
            // We've made a vector in a valid direction, use it.
            proposed_vector = new_vector;
        }
        proposed_vector.normalize();
        proposed_vector *= length;
        proposed_position = proposed_vector + centre;
    }
    else // Too short.
    {
        // Only change the position along the direction of normal.
        // There will be two valid position along this line, we pick the
        // farthest one in the direction of the normal.
        
        // To calculate the distance to the sphere we need to change the
        // vector's basis to a orthonormal basis where one of the components
        // is in the direction of the normal. We can then replace this
        // component with the distance to the sphere, having calculated where
        // it is in the other two directions.
        
        // For example, let the basis B be contain vectors S, T, normal.
        // If proposed_vector can be expressed as s * S + t * T + n * normal
        // then we replace n with sqrt(length * length - s * s - t* t).
        // Then we convert the vector back to the standard basis.
        
        // Find basis S, T, normal:
        btVector3 S = normal.cross(btVector3(0.0, 0.0, 1.0));
        if (S.length2() < 0.25)
        {
            // Try another direction for numerical stablility.
            S = normal.cross(btVector3(0.0, 1.0, 0.0));
        }
        btVector3 T = normal.cross(S);
        T.normalize();
        S = T.cross(normal);
        // Find the change of basis coordinates of proposed_vector in S & T.
        btScalar s = proposed_vector.dot(S);
        btScalar t = proposed_vector.dot(T);
        // find ideal distance
        btScalar n = std::sqrt(length * length - s * s - t * t);
        // now convert back into the standard basis.
        proposed_vector = s * S + t * T + n * normal;
        proposed_position = centre + proposed_vector;
    }
}

void RotationHandle::draw() const
{
    // line to handle
#ifndef HAVE_GLES
    glBegin(GL_LINES);
        glVertex3f(centre.x(), centre.y(), centre.z());
        glVertex3f(position.x(), position.y(), position.z());
    glEnd();
    
    // blob on the end.
    glBegin(GL_POINTS);
        glVertex3f(position.x(), position.y(), position.z());
    glEnd();
#endif
    
    glPushMatrix();
        glTranslatef(position.x(), position.y(), position.z());
        btScalar scale = position.distance(centre);
        glScalef(scale, scale, scale);
        btVector3 difference = (position - centre).normalized();
        btVector3 cross = difference.cross(btVector3(0.0, 0.0, 1.0)).cross(difference).normalized();
        btVector3 other = cross.cross(difference);
        GLfloat mat[16] = {cross.x(), cross.y(), cross.z(), 0,
                           other.x(), other.y(), other.z(), 0,
                           difference.x(), difference.y(), difference.z(), 0,
                           0, 0, 0, 1};
        glMultMatrixf(mat);
        float angles[12];
        for (int i = 0; i < 12; i++)
        {
            angles[i] = -0.392699082 + (float (i)) * 0.065449847;
        }
#ifndef HAVE_GLES
        glBegin(GL_LINE_STRIP);
            for (int i = 0; i < 12; i++)
            {
                const float & angle = angles[i];
                glVertex3f(std::sin(angle), 0.0, std::cos(angle) - 1.0);
            }
        glEnd();
        glBegin(GL_LINE_STRIP);
            for (int i = 0; i < 12; i++)
            {
                const float & angle = angles[i];
                glVertex3f(0.0, std::sin(angle), std::cos(angle) - 1.0);
            }
        glEnd();
#endif
    glPopMatrix();
}

void RotationHandle::set_length(btScalar new_length)
{
    length = new_length;
}

void RotationHandle::set_centre(btVector3 new_centre)
{
    centre = new_centre;
}

}

}
