/** @file libtrack/edit_base/LineConstrainedControlPoint.h
 *  @brief Declare the Track::EditAssist::LineConstrainedControlPoint class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "LineConstrainedControlPoint.h"

#include <GL/gl.h>
#include <cmath>

namespace Track
{

namespace EditAssist
{

LineConstrainedControlPoint::LineConstrainedControlPoint()
    :   allow_before_start(true)
    ,   allow_after_end(true)
{
}

LineConstrainedControlPoint::LineConstrainedControlPoint(bool allow_before_start, bool allow_after_end)
    :   allow_before_start(allow_before_start)
    ,   allow_after_end(allow_after_end)
{
}

LineConstrainedControlPoint::~LineConstrainedControlPoint()
{
}

/** snap a point to a plane going through the origin.
 * @param v position to snap
 * @param normal The normal of the plane
 * @return the nearest point to v on the plane.
 */
btVector3 to_plane(btVector3 v, btVector3 normal)
{
    btScalar height = v.dot(normal);
    return v - height * normal;
}

void LineConstrainedControlPoint::snap(btVector3 & position, btVector3 normal) const
{
    // find nearest point in screenspace to line.
    btVector3 position_flat = to_plane(position, normal);
    btVector3 start_flat = to_plane(start, normal);
    btVector3 stop_flat = to_plane(stop, normal);
    // find distance along line.
    btVector3 line = stop_flat - start_flat;
    btVector3 to_start = position_flat - start_flat;
    btScalar line_length_squared = line.length2();
    if (line_length_squared  == 0)
    {
        // All points are equally close in this view. Pick any valid position.
        position = start;
        return;
    }
    btScalar u = to_start.dot(line) / line_length_squared;
    // is it outside the allowed range?
    if (u < 0.0 && !allow_before_start)
    {
        position = start;
        return;
    }
    else if (u > 1.0 && !allow_after_end)
    {
        position = stop;
        return;
    }
    else
    {
        // use u to find the position in 3D.
        position = start + (stop - start) * u;
    }
}

void LineConstrainedControlPoint::draw() const
{
#ifndef HAVE_GLES
    glBegin(GL_LINES);
        glVertex3f(start.x(), start.y(), start.z());
        glVertex3f(position.x(), position.y(), position.z());
    glEnd();
    glBegin(GL_POINTS);
        glVertex3f(position.x(), position.y(), position.z());
    glEnd();
#endif
    glPushMatrix();
        glTranslatef(position.x(), position.y(), position.z());
        btScalar scale = position.distance(start) / 4.0;
        glScalef(scale, scale, scale);
        btVector3 difference = (position - start).normalized();
        btVector3 cross = difference.cross(btVector3(0.0, 0.0, 1.0)).cross(difference).normalized();
        btVector3 other = cross.cross(difference);
        GLfloat mat[16] = {cross.x(), cross.y(), cross.z(), 0,
                           other.x(), other.y(), other.z(), 0,
                           difference.x(), difference.y(), difference.z(), 0,
                           0, 0, 0, 1};
        glMultMatrixf(mat);
#ifndef HAVE_GLES
        glBegin(GL_LINE_LOOP);
            for (int i = 0; i < 13; i++)
            {
                float angle = (float (i)) * 0.523598776;
                glVertex2f(std::sin(angle), std::cos(angle));
            }
        glEnd();
#endif
    glPopMatrix();
}

} // namespace EditAssist

} // namespace Track
