/** @file libtrack/edit_base/EdgeStrengthHandle.h
 *  @brief Declare the Track::EditAssist::EdgeStrengthHandle class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_EDIT_BASE_EDGE_STRENGTH_HANDLE_H
#define LIBTRACK_EDIT_BASE_EDGE_STRENGTH_HANDLE_H

#include <LinearMath/btQuaternion.h>

#include "LineConstrainedControlPoint.h"

namespace Track
{

namespace EditAssist
{

/** A handle for adjusting the gradient stength of a Track::PathEdge.
 * There is a handle for each end. It controls where the beizer control point
 * that doesn't neceserially meet the edge is. The farther out the handle is
 * from the vertex, the more the vertices angle affects the path.
 * 
 * We do not allow any placement of the control point, as the gradient at the
 * vertex must be correct for the segments to line up. It is constrained along
 * a line.
 */
class EdgeStrengthHandle
    :   public LineConstrainedControlPoint
{
public:
    enum EdgeEnd
    {
        EE_SOURCE = 0 /// The end with the in node.
        , EE_DESTINATION = 1 /// The edge with the out node.
    };
    /** Create linked to a specific edge.
     * @param edge_index The name of the edge to link to.
     * @param adjust_end The end of the edge the handle adjusts.
     */
    EdgeStrengthHandle(std::size_t edge_index, EdgeEnd adjust_end);
    virtual ~EdgeStrengthHandle();
    
    /** Update the position of the line.
     * Should be called when the connected vertex moves or rotates. This needs
     * to be called before snap().
     */
    void update(btVector3 position, btQuaternion angle, btScalar length);
    
    virtual boost::shared_ptr<Document::DocumentDelta> make_delta(btVector3 position) const;
    
    virtual void draw() const;
protected:
    std::size_t edge_index;
    EdgeEnd adjust_end;
};

} // EditAssist namespace

} // Track namespace

#endif // LIBTRACK_EDIT_BASE_EDGE_STRENGTH_HANDLE_H
