/** @file libtrack/edit_base/EdgeStrengthHandle.cpp
 *  @brief Implement the Track::EditAssist::EdgeStrengthHandle class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "EdgeStrengthHandle.h"

#include <GL/gl.h>
#include <LinearMath/btTransform.h>

#include "../document/SetEdgeStrengthDelta.h"

namespace Track
{

namespace EditAssist
{

EdgeStrengthHandle::EdgeStrengthHandle(std::size_t edge_index, EdgeEnd adjust_end)
    :   LineConstrainedControlPoint(false, true)
    ,   edge_index(edge_index)
    ,   adjust_end(adjust_end)
{
}

EdgeStrengthHandle::~EdgeStrengthHandle()
{
}

void EdgeStrengthHandle::update(btVector3 source, btQuaternion angle, btScalar length)
{
    start = source;
    
    // ensure the line used for constraint calculation never has 0 length,
    // even when the displayed line has 0 length.
    stop = source + btTransform(angle)(btVector3(0.0, 1.0, 0.0));
    position = source + btTransform(angle)(btVector3(0.0, length, 0.0));
}

boost::shared_ptr<Document::DocumentDelta> EdgeStrengthHandle::make_delta(btVector3 position) const
{
    // Find strength. Position is always along the line.
    btVector3 line = position - start;
    btScalar strength = line.length();
    return boost::shared_ptr<Document::DocumentDelta>
    (
        new Document::SetEdgeStrengthDelta
        (
            Document::EdgeStrengthFinder(edge_index, adjust_end),
            strength
        )
    );
}

void EdgeStrengthHandle::draw() const
{
    glColor4f(0.5,0.5,1.0,1.0);
    LineConstrainedControlPoint::draw();
}

} // namespace EditAssist

} // namespace Track
