/** @file document/StackableDelta.h
 *  @brief Declare the Document::StackableDelta class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_DOCUMENT_STACKABLE_DELTA_H_
#define LIBTRACK_DOCUMENT_STACKABLE_DELTA_H_

#include <vector>

#include <boost/shared_ptr.hpp>

#include "DocumentDelta.h"


namespace Document
{

/** A DocumentDelta that will combine the effects of other DocumentDeltas.
 * You can stack document deltas up in the apply() function using
 * add_to_stack(delta, track).
 */
class StackableDelta
    :   public DocumentDelta
{
public:
    StackableDelta();
    virtual ~StackableDelta();
    
    /** Create a stack of DocumentDeltas.
     * Use only add_to_stack() to change the track, so that changes can be
     * reversed.
     * @param track The track to apply the changes to.
     */
    virtual void apply(Track::Track & track) = 0;
    /** Undo the stack of DocumentDeltas created in apply().
     */
    void unapply(Track::Track & track);
protected:
    /** Apply and record a change.
     * This will perform the apply command for the DocumentDelta you pass in,
     * and record it so that it can be undone later.
     * @param delta shared pointer to the DocumentDelta to apply.
     * @param track The Track::Track to apply it to.
     */
    void add_to_stack(boost::shared_ptr<DocumentDelta> delta, Track::Track & track);
private:
    /// The DocumentDeltas currently ready to be undone, last first.
    std::vector<boost::shared_ptr<DocumentDelta> > delta_stack;
};

} // Document namespace

#endif // LIBTRACK_DOCUMENT_STACKABLE_DELTA_H_
