/** @file document/SetStartEdgeDelta.h
 *  @brief Implement the Document::SetStartEdgeDelta,
 *  Document::StartEdgeFinder, and Document::SetStartPositionsDelta classes.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "SetStartEdgeDelta.h"

#include "InsertTrackAttachmentDelta.h"
#include "../Track.h"
#include "../StartingPosition.h"

namespace Document
{

StartEdgeFinder::StartEdgeFinder()
{
}

void StartEdgeFinder::read(const Track::Track & track, unsigned long int & edge_name)
{
    edge_name = track.get_path().get_starting_edge();
}

void StartEdgeFinder::write(Track::Track & track, const unsigned long int & new_edge_name)
{
    track.get_path().set_starting_edge(new_edge_name);
}

SetStartPositionsDelta::SetStartPositionsDelta(const unsigned int edge_name)
    :   m_edge_name(edge_name)
{
}

void SetStartPositionsDelta::apply(Track::Track & track)
{
    // deltas to delete the Track::StartMarkers on the current edge.
    unsigned int old_edge_name = track.get_path().get_starting_edge();
    if (old_edge_name != 0)
    {
        const Track::PathEdge &old_edge = track.get_path().get_edge(old_edge_name);
        std::vector<boost::shared_ptr<Track::TrackAttachment>
                   >::const_iterator it = old_edge.get_attachments().begin();
        while (it != old_edge.get_attachments().end())
        {
            if (dynamic_cast<const Track::StartingPosition *>(&(**it)))
            {
                boost::shared_ptr<Document::DocumentDelta> delta
                (
                    new Document::RemoveTrackAttachmentDelta
                    (
                        InsertRemoveTrackAttachmentIdentifier(*it)
                    )
                );
                add_to_stack(delta, track);
                // Invalidates iterator, start again.
                it = old_edge.get_attachments().begin();
            } else {
                it++;
            }
        }
    }
    // delta to replace the starting edge
    add_to_stack(boost::shared_ptr<DocumentDelta>(
                    new SetStartEdgeDelta(Document::StartEdgeFinder(),
                                          m_edge_name)
               ), track);
    
    // Deltas to add new start markers.
    for (int i = 0; i < 30; i++)
    {
        boost::shared_ptr<Track::TrackAttachment> pos(new Track::StartingPosition(29-i));
        pos->set_lateral_position((i % 2 - 0.5) * 1.25);
        pos->set_t_position(0.03125 * i + 0.0625);
        pos->edge_name = m_edge_name;
        
        boost::shared_ptr<Document::DocumentDelta> delta
        (
            new Document::InsertTrackAttachmentDelta(pos)
        );
        add_to_stack(delta, track);
    }
}

} // namespace Document
