/** @file document/RotateVertexDelta.h
 *  @brief Declare the Document::RotateVertexDelta and Document::VertexAngleFinder classes.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef ROTATE_VERTEX_DELTA_H_
#define ROTATE_VERTEX_DELTA_H_

#include <cstddef>

#include <LinearMath/btQuaternion.h>

#include "ChangePropertyDelta.h"

namespace Document
{

/** Find/Change the angle of a particular vertex.
 */
class VertexAngleFinder
    :   public PropertyDeltaIdentifier<btQuaternion>
{
public:
    /** Create associating with a vertex.
     * @param vertex_id The name of the vertex to associate with.
     */
    VertexAngleFinder(std::size_t vertex_id);
    /** Read the assiociated vertex's angle.
     * @param track The track containing the associated vertex.
     * @param angle Variable to set to the angle of the associated vertex.
     */
    virtual void read(const Track::Track & track, btQuaternion & angle);
    /** Change the associated vertex's angle.
     * @param track The track containing the associated vertex.
     * @param new_angle The angle to set the associated vertex to. After
     * rotating, the positive z direction is used as up; and the positive y
     * direction is generally forward, along the track.
     */
    virtual void write(Track::Track & track, const btQuaternion & new_angle);
protected:
    /// The name of the associated vertex.
    std::size_t vertex_id;
};

/** Command for changing the angle of a PathVertex in a Path.
*/
typedef ChangePropertyDelta<VertexAngleFinder, btQuaternion> RotateVertexDelta;

}

#endif // ROTATE_VERTEX_DELTA_H_
