/** @file libtrack/document/RemoveDelta.h
 *  @brief Declare the Document::RemoveDelta class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_DOCUMENT_REMOVE_DELTA_H
#define LIBTRACK_DOCUMENT_REMOVE_DELTA_H

#include <boost/shared_ptr.hpp>

#include "DocumentDelta.h"

namespace Document
{

/** Class to remove something from a track.
 * If removing an object also requires the alteration of dependant objects,
 * make a Document::StackableDelta to find and remove the dependents, then
 * apply a Document::RemoveDelta.
 * @tparam InsertRemoveIdentifer class to identify an object and insert or
 * remove it. Should behave like a Document::InsertRemoveIdentifier.
 * @tparam Object the class of object that is added and removed. Must be
 * assignable.
 */
template <class InsertRemoveIdentifer, class Object>
class RemoveDelta
    :   public DocumentDelta
{
public:
    /** Create a delta instructing the removal of the object found by identifier.
     */
    RemoveDelta(InsertRemoveIdentifer identifier)
        :   identifier(identifier)
    {
    }
    
    virtual ~RemoveDelta()
    {
    }
    
    /// Remove the object found by the identifier.
    virtual void apply(Track::Track & track)
    {
        // remove, but copy the object to restore it if wanted later.
        object = boost::shared_ptr<Object>(new Object(identifier.remove(track)));
    }
    
    /// Add the object back again.
    virtual void unapply(Track::Track & track)
    {
        identifier.insert(track, *object);
    }
protected:
    boost::shared_ptr<Object> object;
    InsertRemoveIdentifer identifier;
};

}

#endif // LIBTRACK_DOCUMENT_REMOVE_DELTA_H
