/** @file document/MoveTrackAttachmentDelta.h
 *  @brief Declare the Document::MoveTrackAttachmentDelta and Document::TrackAttachmentPositionFinderclass. 
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef DOCUMENT_MOVE_TRACK_ATTACHMENT_DELTA_H
#define DOCUMENT_MOVE_TRACK_ATTACHMENT_DELTA_H

#include "ChangePropertyDelta.h"

#include <LinearMath/btVector3.h>
#include <cstddef>

namespace Document
{

/** Find or set the position of a vertex.
 */
class TrackAttachmentPositionFinder
    :   public PropertyDeltaIdentifier<btVector3>
{
    public:
        /** Create, associating with a vertex.
         * @param edge_name The name of the edge with the attachment on it.
         * @param attachment_nmae The name of the attachment.
         */
        TrackAttachmentPositionFinder(std::size_t edge_name, std::size_t attachment_name);
        
        /** Get the location of the associated track attachment.
         * @param track The Track the associated vertex is in.
         * @param position Position of the TrackAttachment. The first
         * component is how far along the edge's curve the point is on the
         * scale of 0 (beginning) to 1 (end). The second component is the 
         * lateral position, and the third component is the vertical position.
         */
        virtual void read(const Track::Track & track, btVector3 & position);
        
        /** Move the associated track attachment.
         * @param track The Track containing the associated vertex.
         * @param new_position The location to move the associated vertex to.
         */
        virtual void write(Track::Track & track, const btVector3 & new_position);
    protected:
        /// The name of the edge.
        std::size_t m_edge_name;
        /// The name of the attachment
        std::size_t m_attachment_name;
};

/** Command for moving control points along the path.
*/
typedef ChangePropertyDelta<TrackAttachmentPositionFinder, btVector3> MoveTrackAttachmentDelta;

} // namespace Document

#endif // DOCUMENT_MOVE_TRACK_ATTACHMENT_DELTA_H
