/** @file libtrack/document/InsertVertexDelta.cpp
 *  @brief Implement the 
 *  Document::InsertRemoveVertexIdentifier,
 *  Document::InsertVertexDelta, and
 *  Document::RemoveVertexDelta classes.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "InsertVertexDelta.h"
#include "../Track.h"
#include "ClearVertexDelta.h"

#include <Debug.h>

namespace Document
{

InsertRemoveVertexIdentifier::InsertRemoveVertexIdentifier(Track::PathVertex object)
    :   InsertRemoveIdentifier<std::size_t, Track::PathVertex>(object.get_name())
{
}

InsertRemoveVertexIdentifier::InsertRemoveVertexIdentifier(std::size_t vertex_index)
    :   InsertRemoveIdentifier<std::size_t, Track::PathVertex>(vertex_index)
{
}

void InsertRemoveVertexIdentifier::insert(Track::Track & track, Track::PathVertex object) const
{
    assert(object.get_name() == identifier);
    object.set_track(&track);
    Track::Path::Graph & graph = track.get_path().graph;
    boost::add_vertex(object, graph);
}

Track::PathVertex InsertRemoveVertexIdentifier::remove(Track::Track & track) const
{
    Track::Path & path = track.get_path();
    Track::Path::Graph & graph = path.graph;
    Track::Path::Graph::vertex_descriptor vertex_descriptor = path.get_node_descriptor(identifier);
    // check that there is no edges using this vertex in debug build.
    assert(boost::out_degree(vertex_descriptor, graph) == 0);
    assert(boost::in_degree(vertex_descriptor, graph) == 0);
    // remove the vertex, but keep the data so we can undo it later.
    Track::PathVertex vertex = graph[vertex_descriptor];
    boost::remove_vertex(vertex_descriptor, graph);
    return vertex;
}

RemoveVertexDelta::RemoveVertexDelta(std::size_t vertex_index)
    :   vertex_index(vertex_index)
{
}

void RemoveVertexDelta::apply(Track::Track & track)
{
    // remove attached edges
    boost::shared_ptr<DocumentDelta> clear_delta(
        new ClearVertexDelta(vertex_index)
    );
    add_to_stack(clear_delta, track);
    
    // now remove the vertex itself.
    boost::shared_ptr<DocumentDelta> delta(new RemoveVertexOnlyDelta(vertex_index));
    add_to_stack(delta, track);
}

} // namespace Document

