/** @file document/Document.h
 *  @brief Declare the Document::Document class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef DOCUMENT_H_
#define DOCUMENT_H_

#include <istream>
#include <vector>

#include <boost/shared_ptr.hpp>
#include <sigc++/sigc++.h>

#include <libtrack/Track.h>

#include "DocumentDelta.h"

/** Document handling for racer_editor.
 * Used for objects that manipulate the track.
 */
namespace Document
{

/** The set of data the editor exists to manipulate.
 *  Anything that responds to open, save, undo, or redo is in a document.
 *  The editor has a single document it works on at once.
 */
class Document
{
public:
    /** Initalise an default document for a given theme.
     * Called when a theme is selected for a new document.
     */
    Document(const Track::Theme & theme);
    
    /** Initalise from a stream.
     * Most likely caused from an open command.
     * This should behave exactly like @code Track::Track(source, theme) @endcode
     * @param source The stream to extract data from. It is advanced to the end
     * of the document.
     * @param theme The theme of the track.
     */
    Document(std::istream & source, const Track::Theme & theme);
    
    virtual ~Document();
    
    
    /** Perform a command
     * This clears any avaliable redos, and adds it to the stack of undo
     * commands.
     * @param delta The command to perform. If 0, no command is performed, but
     * any previewed command is undone.
     */
    void do_command(boost::shared_ptr<DocumentDelta> delta);
    
    /** Show the effect of a command for a preview.
     * This performs a command, but it is automatically undone later. It is
     * useful for showing the effect of a command while varying parameters.
     * 
     * The command is performed immediately, but undone if any of these
     * functions are performed: do_command(), undo_command(), redo_command(),
     * preview_command(), cancel_preview(), and save().
     * 
     * To accept the change, call do_command() with the final delta.
     * To cancel undo it with no other side effects, call cancel_preview().
     * 
     * The redo history is not lost when using preview_command().
     * 
     * @param delta The command to perform. If 0, no command is performed, but
     * the previous preview may be undone.
     */
    void preview_command(boost::shared_ptr<DocumentDelta> delta);
    
    /** Undo the last preview_command().
     * This action is also performed when do_command(),  undo_command(),
     * redo_command(), preview_command(), or save() are used.
     */
    void cancel_preview();
    
    /// Undo the last command done.
    void undo_command();
    
    /// Redo a command that was previously undone.
    void redo_command();
    
    /** Signal that is emitted whenever a command is run.
     * This incudes when a command is undone.
     * Use this to update the display in the editor.
     * The prototype function for the handler is: 
     * @code void on_command_run(); @endcode
     */
    sigc::signal<void> signal_command_run();
     
    /** Check if there are commands to be undone.
     * @return @code true @endcode if a command can be undone, @code false
     * @endcode if all commands have already been undone.
     */
    bool get_undo_avaliable() const;
    
    /** Check if there are commands to be redone.
     * @return @code true @endcode if a command can be redone, @code false
     * @endcode if there are no commands to be redone.
     */
    bool get_redo_avaliable() const;
    
    /// Save the document.
    void save(std::ostream & destination);
    
    /** Is the document in the state it was saved in?
     * @return true if the document is in the state it was saved in, false if it
     * has been modified.
     */
    bool is_saved() const;
    
    /** Return the number of commands done (or undone) since the last save.
     */
    unsigned int get_changes_since_save() const;
    
    /// Return the track so it can be displayed.
    const Track::Track & get_track() const;
protected:
    /// The track in the document in its current state.
    Track::Track track;
    
    /// Stack of commands that have been done
    std::vector<boost::shared_ptr<DocumentDelta> > done;
    
    /// Stack of commands that have been undone
    std::vector<boost::shared_ptr<DocumentDelta> > undone;
    
    /// Signal to emit when a delta is applied or unapplied.
    sigc::signal<void> m_signal_command_run;
    
    /** Count undo / redo history to the point where the file is saved.
     * For every action undone, this is decrement, for every action redone, it
     * is incremented. If it is 0 and @code saved_in_history @endcode is
     * @code true @endcode , the document is in the saved state.
     */
    signed int commands_from_save;
    
    /** True if the state when the document was saved is stored in the undo /
     * redo history.
     */
    bool saved_in_history;
    
    /** Command to preview the effects of.
     * If 0, we are not previewing any command.
     * Otherwise, we should undo the command before doing anything else.
     */
    boost::shared_ptr<DocumentDelta> temporary_command;
};

}

#endif // DOCUMENT_H_
