/** @file document/Document.cpp
 *  @brief Implement the Document::Document class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "Document.h"

#include <istream>
#include <vector>

#include <libtrack/Track.h>

namespace Document
{

Document::Document(const Track::Theme & theme)
    :   track(theme, true)
    ,   commands_from_save(0)
    ,   saved_in_history(false)
{
}

Document::Document(std::istream & source, const Track::Theme &theme)
    :   track(source, theme, true)
    ,   commands_from_save(0)
    ,   saved_in_history(true)
{
}

Document::~Document()
{
}

void Document::do_command(boost::shared_ptr<DocumentDelta> delta)
{
    cancel_preview();
    if (!delta) return;
    delta->apply(track);
    done.push_back(delta);
    // We can't redo events in the queue, since they depend on this command not
    // having been done.
    undone.clear();
    commands_from_save++;
    if (commands_from_save <= 0)
    {
        // We've cleared the redo history that included the saved state.
        saved_in_history = false;
    }
    m_signal_command_run.emit();
}

void Document::preview_command(boost::shared_ptr<DocumentDelta> delta)
{
    cancel_preview();
    if (!delta) return;
    temporary_command = delta;
    delta->apply(track);
    m_signal_command_run.emit();
}
    
void Document::cancel_preview()
{
    if (temporary_command)
    {
        temporary_command->unapply(track);
        m_signal_command_run.emit();
        temporary_command = boost::shared_ptr<DocumentDelta>();
    }
}

void Document::undo_command()
{
    cancel_preview();
    if (!done.empty())
    {
        done.back()->unapply(track);
        undone.push_back(done.back());
        done.pop_back();
        commands_from_save--;
        
        m_signal_command_run.emit();
    }
}

void Document::redo_command()
{
    cancel_preview();
    if (!undone.empty())
    {
        undone.back()->apply(track);
        done.push_back(undone.back());
        undone.pop_back();
        commands_from_save++;
        m_signal_command_run.emit();
    }
}

sigc::signal<void> Document::signal_command_run()
{
    return m_signal_command_run;
}

bool Document::get_undo_avaliable() const
{
    return !done.empty();
}
 
bool Document::get_redo_avaliable() const
{
    return !undone.empty();
}

void Document::save(std::ostream & destination)
{
    cancel_preview();
    // Write track information to output stream.
    destination << track;
    commands_from_save = 0;
    saved_in_history = true;
}

bool Document::is_saved() const
{
    return saved_in_history && (commands_from_save == 0);
}

const Track::Track & Document::get_track() const
{
    return track;
}

unsigned int Document::get_changes_since_save() const
{
    if (commands_from_save > 0) return unsigned(commands_from_save);
    return unsigned(-commands_from_save);
}

}
