/** @file document/ChangeVertexSegmentDelta.h
 *  @brief Declare the Document::ChangeVertexSegmentDelta class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef CHANGE_VERTEX_SEGMENT_DELTA_H_
#define CHANGE_VERTEX_SEGMENT_DELTA_H_

#include <cstddef>

#include "ChangePropertyDelta.h"
#include "StackableDelta.h"

namespace Document
{

/** Find/Change the segment associated with a particular vertex.
 */
class VertexSegmentFinder : public PropertyDeltaIdentifier<std::size_t>
{
public:
    /** Create associating with a vertex.
     * @param node_id The name of the vertex to associate with.
     */
    VertexSegmentFinder(std::size_t node_id);
    /** Read the segment index
     * Reads the segment index of the associated vertex.
     * @param track The track containing the associated vertex.
     * @param segment_index This will be set the assicated vertex's segment's
     * index.
     */
    virtual void read(const Track::Track & track, std::size_t & segment_index);
    /** Change the associated vertex's segment.
     * @param track The track containing the associated vertex.
     * @param new_segment_index The index of the segment to use.
     */
    virtual void write(Track::Track & track, const std::size_t & new_segment_index);
    
    /// Get the associated vertex's name passed to the constructor.
    std::size_t get_vertex_name() const;
protected:
    /// The name of the associated vertex.
    std::size_t node_id;
};

/** Command for switching the segment at a vertex in a path.
 * Does not remove associated edges, so it isn't safe to use this on arbitrary
 * vertices.
*/
typedef ChangePropertyDelta<VertexSegmentFinder, std::size_t> ChangeVertexSegmentOnlyDelta;

/** Command for switching the segment at a vertex in a path.
 * This removes any edges attached to the vertex, so there is no danger
 * of dangling SegmentConnections.
*/
class ChangeVertexSegmentDelta
    :   public StackableDelta
{
public:
    /** Create delta for changing the segment at a vertex.
     * @param vertex_segment_finder identifies which vertex to change.
     * @param new_segment_index The index of the Track::Segment in the
     * Track::Theme to switch to.
     */
    ChangeVertexSegmentDelta(VertexSegmentFinder vertex_segment_finder, std::size_t new_segment_index);
    
    virtual void apply(Track::Track & track);
protected:
    VertexSegmentFinder vertex_segment_finder;
    std::size_t new_segment_index;
};

}

#endif // CHANGE_VERTEX_SEGMENT_DELTA_H_
