/** @file document/ChangePropertyDelta.h
 *  @brief Declare the Document::ChangePropertyDelta class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef CHANGE_PROPERTY_DELTA_H_
#define CHANGE_PROPERTY_DELTA_H_

#include "DocumentDelta.h"

namespace Document
{

/** Virtual base class to help with the Identifer template parameter of
 * ChangePropertyDelta.
 * It is not necessary to inherit from this class, but you should write
 * a class with read and write properties that do the same thing.
 */
template<class Data>
class PropertyDeltaIdentifier
{
public:
    virtual ~PropertyDeltaIdentifier()
    {
    }
    /** Read the data from track and write it to data.
     * @param track The track to extract the data from.
     * @param data The variable to store the data too.
     */
    virtual void read(const Track::Track & track, Data & data) = 0;
    /** Write the data to track.
     * @param track The track to write the data to.
     * @param data The data to write.
     */
    virtual void write(Track::Track & track, const Data & data) = 0;
};

/** Generic DocumentDelta for changing a property without adding or
 * deleting data.
 * @tparam Identifier A class that finds an editable property.
 * @tparam Data The class of the editable property.
 */
template <class Identifier, class Data>
class ChangePropertyDelta
    :   public DocumentDelta
{
public:
    /** Command changes the information found by Identifier to Data.
     *  @param identifier Object that finds the data that will be changed. It
     *  must behave like a PropertyDeltaIdentifier, but does not need to
     *  inherit from it.
     *  @param data The new replacement to the data specified by identifier.
     */
    ChangePropertyDelta(Identifier identifier, Data data)
        :   identifier(identifier)
        ,   new_data(data)
    {
    }
    virtual void apply(Track::Track & track)
    {
        identifier.read(track, old_data);
        identifier.write(track, new_data);
    }
    virtual void unapply(Track::Track & track)
    {
        identifier.write(track, old_data);
    }
protected:
    Identifier identifier;
    Data new_data;
    Data old_data;
};

}

#endif
