/** @file libtrack/UniqueIdentifier.h
 *  @brief Declare the Track::UniqueIdentifier class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_UNIQUE_IDENTIFIER_H_
#define LIBTRACK_UNIQUE_IDENTIFIER_H_

namespace Track
{

/** Class that gives its instances unique names.
 * It is used instead of pointers so that an object may be moved, copied, freed 
 * and reallocated and the names should act intelligently, for example in undo /
 * redo names are preserved, but in copy / paste they are not.
 * 
 * When you copy-construct or assign a UniqueIdentifier instance, the source
 * object's name is kept. This is useful for operations such as undo and redo,
 * where a copy of a previously deleted object can be put back with the same
 * name. However, sometimes a new name is wanted (for example when copy / paste
 * is used, or you are creating a new object similar to a previous one and you 
 * want to keep using both copys.) In this case, make sure to use the default
 * constuctor instead of the copy constructor (you may need a special
 * constuctor).
 */
class UniqueIdentifier
{
public:
    /// Create an object with a name unique to objects of the same class.
    UniqueIdentifier()
    {
        static unsigned long int static_name = 0;
        my_name = static_name++;
    }
    
    /// Get the assigned name.
    unsigned long int get_name() const
    {
        return my_name;
    }
private:
    /// name assigned to this object
    unsigned long int my_name;
};

}

#endif
