/** @file libtrack/TrackMesh.h
 *  @brief Declare the Track::TrackMesh class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_TRACKMESH_H_
#define LIBTRACK_TRACKMESH_H_

#include "Mesh/MeshFaces.h"
#include "Mesh/MultiDrawableMesh.h"
#include "PieceDistortion.h"
#include "SegmentConnection.h"
#include <Debug.h>

#include <vector>
#include <istream>
#include <limits>

#include <boost/shared_ptr.hpp>

namespace Track
{

/** A mesh that can be chained together and distorted to make a track with
 * corners and twists.
 * A theme contains a variety of Segments for different looks and purposes,
 * which define a few TrackMeshes each (for physics, AI, graphics).
 * 
 * A TrackMesh is linked to from TrackMeshInstances.
 * @tparam T class to use as the mesh. Should behave like MeshFaces.
 */
template <class T = MeshFaces>
class TrackMesh
{
public:
    /** Create from a description in a stream.
     * Advances the stream to the end of the TrackMesh.
     */
    TrackMesh(std::istream & source);
    
    virtual ~TrackMesh();
    
    /** The sharpest corner in radians this track mesh can manage.
     * It is determined algorithmically from the mesh's shape.
     */
    btScalar get_maximum_bend() const;
    
    /** Get the length of the segment.
     */
    btScalar get_length() const;
    
    /** Get the minimum y value of all vertices in the mesh.
     */
    btScalar get_minimum_y() const;
    
    /** get the faces of a distorted mesh.
     * This is useful for distorting a navigation mesh for AI.
     */  
    T get_distorted_faces(PieceDistortion piece_distortion) const;
    
    /** get the faces of a linearly transformed mesh.
     * This is useful for getting collision shapes.
     * 
     * Graphics meshes should use get_faces(), and do transformations in the
     * graphics pipeline instead for better memory use.
     */  
    T get_distorted_faces(btTransform tranform) const;
    
    /** get non-distorted mesh.
     */
    const T & get_faces() const;
protected:
    /** The maximum bend a PieceDistortion should have when applied to this
     * mesh.
     * Any more and the vertices of the mesh will cross the wrong way. 
     */
    float maximum_bend;
    
    /// length between end connections (may not be applicable).
    float length;
    
    /// The lowest y value of all vertices' positions in the mesh.
    btScalar minimum_y;
    
    /// The heighest y value of all vertices' positions in the mesh.
    btScalar maximum_y;
    
    /// The mesh with no distortion.
    T mesh;
};

template <class T>
TrackMesh<T>::TrackMesh(std::istream & source)
    :   minimum_y(std::numeric_limits<btScalar>::max())
    ,   maximum_y(std::numeric_limits<btScalar>::min())
    ,   mesh(source)
{
    // Calculate the length.
    minimum_y = mesh.get_bounds().get_min().y();
    maximum_y = mesh.get_bounds().get_max().y();
    length = maximum_y - minimum_y;
    /// @todo calculate the maximum bend.
    DEBUG_MESSAGE("Loaded Track::TrackMesh");
}

template <class T>
TrackMesh<T>::~TrackMesh()
{
}

template <class T>
btScalar TrackMesh<T>::get_maximum_bend() const
{
    return maximum_bend;
}

template <class T>
btScalar TrackMesh<T>::get_length() const
{
    return length;
}

template <class T>
btScalar TrackMesh<T>::get_minimum_y() const
{
    return minimum_y;
}

template <class T>
inline
T TrackMesh<T>::get_distorted_faces(PieceDistortion piece_distortion) const
{
    T result;
    for (unsigned int vert_index = 0;
         vert_index < mesh.get_num_vertices();
         vert_index++)
    {
        result.add_vertex(piece_distortion(mesh.get_vertex_pos(vert_index)));
    }
    // now copy the faces.
    result.copy_faces(mesh);
    return result;
}

// specialization for MultiDrawableMesh
template <>
inline
MultiDrawableMesh TrackMesh<MultiDrawableMesh>::get_distorted_faces(PieceDistortion piece_distortion) const
{
    MultiDrawableMesh result;
    for (unsigned int lod = 0;
         lod < mesh.get_num_levels();
         lod++)
    {
        DrawableMesh out_mesh_level;
        const DrawableMesh &in_mesh_level = mesh.get_level(lod);
        for (unsigned int vert_index = 0;
             vert_index < in_mesh_level.get_num_vertices();
             vert_index++)
        {
            out_mesh_level.add_vertex(piece_distortion(in_mesh_level.get_vertex_pos(vert_index)));
        }
        // now copy the faces.
        out_mesh_level.copy_faces(in_mesh_level);
        // and copy the entire thing to the group of mesh levels.
        result.add_mesh(out_mesh_level, mesh.get_tex_level(lod));
    }
    return result;
}

template <class T>
const T & TrackMesh<T>::get_faces() const
{
    return mesh;
}

template <class T>
inline
T TrackMesh<T>::get_distorted_faces(btTransform transform) const
{
    T result;
    for (unsigned int vert_index = 0;
         vert_index < mesh.get_num_vertices();
         vert_index++)
    {
        result.add_vertex(transform(mesh.get_vertex_pos(vert_index)));
    }
    // now copy the faces.
    result.copy_faces(mesh);
    return result;
}

// specialization for MultiDrawableMesh
template <>
inline
MultiDrawableMesh TrackMesh<MultiDrawableMesh>::get_distorted_faces(btTransform transform) const
{
    MultiDrawableMesh result;
    for (unsigned int lod = 0;
         lod < mesh.get_num_levels();
         lod++)
    {
        DrawableMesh out_mesh_level;
        const DrawableMesh &in_mesh_level = mesh.get_level(lod);
        for (unsigned int vert_index = 0;
             vert_index < in_mesh_level.get_num_vertices();
             vert_index++)
        {
            out_mesh_level.add_vertex(transform(in_mesh_level.get_vertex_pos(vert_index)));
        }
        // now copy the faces.
        out_mesh_level.copy_faces(in_mesh_level);
        // and copy the entire thing to the group of mesh levels.
        result.add_mesh(out_mesh_level, mesh.get_tex_level(lod));
    }
    return result;
}

}

#endif /*TRACKMESH_H_*/
