/** @file libtrack/TrackAttachment.h
 *  @brief Declare the Track::TrackAttachment class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_TRACK_ATTACHMENT_H_
#define LIBTRACK_TRACK_ATTACHMENT_H_

#include <iostream>
#include <LinearMath/btScalar.h>
#include <LinearMath/btTransform.h>
#include "UniqueIdentifier.h"
#include "ClassLoader.h"
#include "NearTrack.h"

namespace Track
{

/** An object in a Track that is locked to the Path.
 * It is fixed onto an edge or vertex in the track editor. When the edge or
 * vertex is moved, it remains fixed to the same point in the track.
 * It attaches to the surface of the AI navigation mesh.
 */
class TrackAttachment
    :   public UniqueIdentifier
    ,   public Savable
{
public:
    TrackAttachment(std::istream & source);
    TrackAttachment();
    virtual ~TrackAttachment();
    
    btScalar get_t_position() const;
    /// Set the position along the edge's curve from 0 to 1.
    void set_t_position(btScalar value);
    
    btScalar get_lateral_position() const;
    /// Set the position along the track perpendicular to the edge's direction.
    void set_lateral_position(btScalar value);
    
    btScalar get_vertical_position() const;
    /** Set the position in the direction of the track's normal.
     * This is for the z direction in car space.
     */
    void set_vertical_position(btScalar value);
    
    /** Write the data needed to reconstruct the object to a stream.
     */
    virtual void add_data(std::ostream & stream) const;
    
    virtual std::string get_class_identifier() const;
    
    std::size_t edge_name;
    /** Draw the attachment.
     * The modelview matrix should be set so that the position using the above
     * and the associated edge is the origin of the modelview matrix.
     */
    virtual void draw() const;
    /** Set the transformation from world space to the attachment's space.
     * This will be used to draw the attachment and locate its handles in the
     * editor.
     * @param pos the new global transformation.
     */
    void set_global_transform(btTransform pos);
    /// Get the transformation from world space to the attachment's space.
    const btTransform & get_global_transform() const;
protected:
    /// The transformation from world space to the attachment's space.
    btTransform m_global_transform;
    /** How far down the path the object is. 
     * Parameter to path's curve, between 0 and 1.
     */
    btScalar m_t_position;
    /** Offset from the centre across the track in spacial units.
     * Positive is to the right in normal direction of travel.
     * 0 is on the curve itself.
     */
    btScalar m_lateral_position;
    /** Offset from the centre perpendicular to the track in spacial units.
     * Positive is above the track in the normal direction travel.
     * 0 is on the curve itself.
     */
    btScalar m_vertical_position;
};

// Add a TrackAttachment or its children to a stream.
std::ostream & operator<<(std::ostream & destination, const TrackAttachment & attachment);

} // namespace Track


#endif // LIBTRACK_TRACK_ATTACHMENT_H_
