/** @file libtrack/Track.h
 *  @brief Declare the Track::Track class.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_TRACK_H_
#define LIBTRACK_TRACK_H_

#include <istream>
#include <vector>
#include <boost/shared_ptr.hpp>

#include "Theme.h"
#include "path/Path.h"
#include "Lighting.h"

#include <BulletCollision/CollisionShapes/btBvhTriangleMeshShape.h>

/** Used in libtrack, a library used by both the game and track editor.
 * Contains the track, theme, and related objects.
 */
namespace Track
{
    
/** A playable stage.
 * A track has a Track::Theme and Paths to drive on.
 * 
 */
class Track
{
    friend std::ostream & operator<<(std::ostream & destination, const Track & track);
public:
    /** Intialise by loading data from a stream, sharing an already loaded
     * theme.
     * @param stream The stream to pull data from. It must point to a track.
     * It will be advanced until the end of the track.
     * @param theme The theme the track was made for.
     * @param editor When set to true, causes the track to act more helpful
     * to the editor. For example, by default tracks are rendered in
     * wireframe.
     */
    Track(std::istream & stream, const Theme & theme, bool editor = false);
    
    /// Make an empty track
    Track(const Theme & theme, bool editor = false);
    
    virtual ~Track();
    
    /// Get a the path without modification.
    const Path & get_path() const;
    
    /// Get the path so it can be changed.
    Path & get_path();
    
    const Theme & get_theme() const;
    
    /// Return the collision shape for the entire track.
    boost::shared_ptr<btCollisionShape> get_collision_shape() const;
    
    /// Return the collision shape for the part of the track you can drive on.
    boost::shared_ptr<btCollisionShape> get_floor_shape() const;
    
    /** Return a mesh of the faces the ai drives on.
     * Faces from different sources that line up will be joined together,
     * allowing the mesh to be used as a navigatable surface.
     * If you have changed the track, the ai mesh does not update
     * automatically. Call update_ai_mesh() to update all references to the
     * ai mesh.
     * Precondition: update_ai_mesh() has been called since the last edit.
     * Postcondition: get_ai_mesh() reflects the surface the AI should consider.
     */
    const MeshFaces & get_ai_mesh() const;
    
    /** Get a graph showing connections between faces that AI can drive on.
     * Call update_ai_mesh() first.
     */
    const MeshFaces::Graph & get_ai_graph() const;
    
    /** Recalculate the surface that the AI can drive on, and its connectivity
     * graph.
     */
    void update_ai_mesh();
    
    /** Return the length of the lap.
     * Requires update_ai_mesh() was called since the last change to the
     * track.
     */
    btScalar get_lap_length() const;
    
    /// Set the filename to record in replay files.
    void set_filename(std::string filename);
    /// Get the filename previously set by set_filename.
    std::string get_filename() const;
    
    const Lighting & get_lighting() const;
    void set_lighting(const Lighting & lighting);
protected:
    /// The vesion used to load the file.
    const unsigned int file_version;
    
    /// The theme of the track.
    const Theme & theme;
    
    /// The path the course follows.
    Path path;
    
    /// If the track should act like in the editor
    bool editor;
    
    /** In editor mode, scan the meshes in the theme and set them to display
     * wireframe.
     */
    void scan_theme();
    
    /** Find position coordinates for the AI mesh.
     * Requires that m_ai_mesh is up to date, but invalidates m_ai_graph.
     * Abuses u texture coordinate to mean how far through the lap a point is.
     * Also sets the m_lap_length.
     */
    void scan_ai_graph();
    
    std::string filename;
    
    /// Mesh used for AI navigation.
    MeshFaces m_ai_mesh;
    /// Graph for AI navigation mesh.
    MeshFaces::Graph m_ai_graph;
    /// Length of shortest path around the track
    btScalar m_lap_length;
    /// The lighting setup of the track.
    Lighting m_lighting;
};

/** write a track to a stream, so it can be loaded with Track(stream)
 * @param destination The stream to write to. If the stream is read back from
 * the position it was in before this function call using
 * @code Track::Track(stream) @endcode , then the resulting Track will be the
 * same as the one passed in.
 * @param track The track to write to a file.
 * @return The same stream reference as destination, however it will have the
 * track written to it when the function completes.
 */
std::ostream & operator<<(std::ostream & destination, const Track & track);

}

#endif /* LIBTRACK_TRACK_H_ */ 
