/** @file libtrack/Theme.h
 *  @brief Declare the Track::Theme class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_THEME_H_
#define LIBTRACK_THEME_H_

#include "Skybox.h"
#include "Segment.h"

#include <vector>
#include <istream>
#include <string>

#include <boost/shared_ptr.hpp>

namespace Track
{

/** A track theme.
 * Includes the sky, the types of track avaiable, and random decorative meshes.
 * It isn't quite swappable across tracks since the themes can define different
 * track segments which have variable constraints to their placement.
 */
class Theme
{
public:
    /** Construct a theme object from an input stream.
     * This will advance the stream to the end of the theme description.
     * 
     * An error may be thrown if the stream is truncated or malformed.
     * @param source A reference to the stream to take construction values from.
     */
    Theme(std::istream & source);
    /// Destruct the theme and free all the pieces.
    virtual ~Theme();
    
    /** get a reference to the skybox object.
     * @return A reference to the skybox that is the background in this theme.
     */
    const Skybox & get_skybox() const;
    
    /** get the number of track segments types avalible for tracks in this theme.
     * @return the number of track segments avaliable.
     */
    std::size_t get_number_of_segments() const;
    
    /** get a segment by index.
     * @param index The of the segment from 0 to one less than the value
     * returned by get_number_of_segments.
     * @return a const reference to the segment requested.
     */
    const Segment & get_segment(std::size_t index) const;
    
    /// get the name of the theme.
    const std::string & get_name() const;
    
    /// Get the index of a segment by name.
    std::size_t get_segment_index(std::string name) const;
    
    /// Get the index of the default segment.
    std::size_t get_default_segment_index() const;
    
    /// Return true iff the theme should use racer's Graphics::SkyParticles.
    bool get_use_sky_particles() const;
protected:
    /// The skybox for the background in this theme.
    Skybox skybox;
    /// The different track segments useable in this theme
    std::vector<boost::shared_ptr<Segment> > segments;
    /// The name of the theme. Will appear as a track property.
    std::string name;
    
    /// True iff the theme should use racer's Graphics::SkyParticles.
    std::size_t m_default_segment_index;
    
    void find_default_segment();
    
    bool m_use_sky_particles;
};

}

#endif /*THEME_H_*/
