/** @file libtrack/Theme.cpp
 *  @brief Implement the Track::Theme class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "Theme.h"
#include <iostream>
#include "stream_loader.h"
#include "FormErrors.h"

#include <Debug.h>

namespace Track
{

Theme::Theme(std::istream & source)
    :skybox(source)
    
{
    fill_vector_from_stream<Segment>(segments, source);
    
    source >> m_use_sky_particles;
    
    find_default_segment();
    
    DEBUG_MESSAGE("Finished loading theme.");
}

Theme::~Theme()
{
    DEBUG_MESSAGE("Theme destroyed.");
}

const Skybox & Theme::get_skybox() const
{
    return skybox;
}

std::size_t Theme::get_number_of_segments() const
{
    return segments.size();
}

const Segment & Theme::get_segment(std::size_t index) const
{
    assert(index < segments.size());
    return *(segments[index]);
}

const std::string & Theme::get_name() const
{
    return name;
}

std::size_t Theme::get_segment_index(std::string name) const
{
    for (std::size_t index = 0;
         index < segments.size();
         index++)
    {
        if (segments[index]->get_name() == name)
        {
            // found it.
            return index;
        }
    }
    // Doesn't exist. The name probably came from a different version of this theme.
    assert(false);
    throw ThemeChangedError();
}

std::size_t Theme::get_default_segment_index() const
{
    return m_default_segment_index;
}

bool Theme::get_use_sky_particles() const
{
    return m_use_sky_particles;
}

void Theme::find_default_segment()
{
    // find a good default segment.
    std::size_t edges_allowed = -1;
    std::size_t called_default = -1;
    std::size_t called_straight = -1;
    std::size_t count = segments.size();
    for (std::size_t i = 0; i < count; i++)
    {
        const Track::Segment & seg = *(segments[i]);
        if (seg.edges_allowed())
        {
            edges_allowed = i;
        }
        if (seg.get_name() == "default" || seg.get_name() == "Default")
        {
            called_default = i;
        }
        if (seg.get_name() == "straight" || seg.get_name() == "Straight")
        {
            called_straight = i;
        }
    }
    if (called_default != std::size_t(-1))
    {
        m_default_segment_index = called_default;
    }
    else if (called_straight != std::size_t(-1))
    {
        m_default_segment_index = called_straight;
    }
    else if (edges_allowed != std::size_t(-1))
    {
        m_default_segment_index = edges_allowed;
    } else {
        // theme shold probably not be allowed, nothing works for edges!
        m_default_segment_index = 0;
    }
}

}
