/** @file libtrack/Texture.h
 *  @brief Declare the Track::Texture class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_TEXTURE_H_
#define LIBTRACK_TEXTURE_H_

#include <string>
#include <istream>
#include <boost/shared_ptr.hpp>
#include <GL/gl.h>

namespace Track
{

/** Wrapper for OpenGL Texture object.
 * Also loads images from a file. Can find the filename from a stream.
 */
class Texture
{
public:
    /// Load filename from a stream
    Texture(std::istream & source);
    /// Take given filename
    Texture(std::string filename);
    Texture();
    virtual ~Texture();
    /// Get a reference to the filename containing the texture's image data.
    const std::string & get_filename() const;
    
    /// Bind the texture so it can be used in OpenGL.
    virtual void bind() const;
    
    /** Load the texture without binding it.
     * Loads the image file into an OpenGL texture.
     * After the texture has been loaded, has no effect.
     * Requires an OpenGL context.
     */
    void make_cache() const;
protected:
    /// filename for the texture.
    std::string m_filename;
    
    /* There are several mutable things.
     * These are used when loading the image the first time.
     * However, they should not affect the the way the object behaves,
     * only prevent disk access and loading the second time around.
     */
    
    /// OpenGL texture name.
    mutable GLuint m_texture_name;
    
    /** Reference counter for the texture name.
     * When there are no more users, the OpenGL texture is deleted using this.
     * The OpenGL texture object remains until all copies of the object that
     * created it have been destroyed.
     */
    mutable boost::shared_ptr<GLuint> m_texture_name_ref;
    
    /// OpenGL texture target, normally GL_TEXTURE_2D.
    mutable GLenum m_target;
    
    /// Number of texels horizontally across the image
    mutable unsigned int m_width;
    /// Number of texels vertically across the image
    mutable unsigned int m_height;
    /// True if the texture has an alpha component, false otherwise.
    mutable bool m_has_alpha;
    
    /** Delete the OpenGL texture object with the given name.
     * This is static to meet boost::shared_ptr requirements.
     * @param texture pointer to the name of the texture object to delete.
     */
    static void delete_gl_texture(GLuint * texture);
    
    /** Create the OpenGL texture obect.
     * @return the name of the texture object. This value is also assigned to
     * m_texture_name.
     */
    GLuint * load_gl_texture() const;
    
    /// Load the file and specify mipmap textures from it.
    virtual void load_from_file() const;
};

}

#endif /*LIBTRACK_TEXTURE_H_*/
