/** @file libtrack/Texture.cpp
 *  @brief Implement the Track::Texture class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "Texture.h"

#include <SDL_image.h>
#include <GL/glu.h>

#include <Debug.h>
#include "stream_loader.h"

namespace Track
{

Texture::Texture(std::istream & source)
    :   m_filename(string_from_stream(source))
    ,   m_texture_name(0)
{
    DEBUG_MESSAGE("Loading Track::TextureBase");
}

Texture::Texture(std::string filename)
    :   m_filename(filename)
    ,   m_texture_name(0)
{
    DEBUG_MESSAGE("Loading Track::TextureBase from texture filename");
}

Texture::Texture()
    :   m_texture_name(0)
{
    DEBUG_MESSAGE("Track::TextureBase default constructor");
}

Texture::~Texture()
{
    DEBUG_MESSAGE("Track::TextureBase destructor");
}

const std::string & Texture::get_filename() const
{
    return m_filename;
}

void Texture::bind() const
{
    if (!m_texture_name)
    {
        m_texture_name_ref = boost::shared_ptr<GLuint>(load_gl_texture(), delete_gl_texture);
    }
    glBindTexture(m_target, m_texture_name);
}

void Texture::make_cache() const
{
    if (!m_texture_name)
    {
        m_texture_name_ref = boost::shared_ptr<GLuint>(load_gl_texture(), delete_gl_texture);
    }
}

void Texture::delete_gl_texture(GLuint * texture)
{
    glDeleteTextures(1, texture);
}

GLuint * Texture::load_gl_texture() const
{
    m_target = GL_TEXTURE_2D;
    glGenTextures(1, &m_texture_name);
    glBindTexture(m_target, m_texture_name);
    
    load_from_file();
    
    return &m_texture_name;
}

void Texture::load_from_file() const
{
    SDL_Surface * image (IMG_Load(m_filename.c_str())); // load the image file
    // check file has loaded OK
    if (!image)
    {
        std::cerr << "Cannot load image file " << m_filename << std::endl;
        throw; 
    }
    
    // get the format of the image.
    m_width = image->w;
    m_height = image->h;
    
    SDL_PixelFormat & format = *(image->format);
    SDL_LockSurface(image);
    if (format.BitsPerPixel == 8)
    {
        /// @todo implement loading paletted images.
        PRINT_STUB_MESSAGE;
    } else {
        // Check if we have an alpha component.
        m_has_alpha = format.Amask;
        // check if we need to swizzle the colour components
        bool swizzle;
        if (m_has_alpha)
        {
            swizzle =    format.Rmask != 0x000000ff
                      || format.Gmask != 0x0000ff00
                      || format.Bmask != 0x00ff0000
                      || format.Amask != 0xff000000;
        } else {
            swizzle =    format.Rmask != 0x0000ff
                      || format.Gmask != 0x00ff00
                      || format.Bmask != 0xff0000;
        }
        /// @todo swizzle.
        if (swizzle)
        {
            PRINT_STUB_MESSAGE;
        }
    } 
    
    // pass to OpenGL.

#if 0//ndef GLES
  gluBuild2DMipmaps(GL_TEXTURE_2D, GL_RGBA, t->w, t->h, GL_RGBA, GL_UNSIGNED_BYTE, t->pixels);
    gluBuild2DMipmaps(m_target,
                      m_has_alpha ? GL_RGBA8 : GL_RGB8,
                      m_width, m_height,
                      m_has_alpha ? GL_RGBA : GL_RGB,
                      GL_UNSIGNED_BYTE,
                      image->pixels);
#else
  glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);
  glTexImage2D(GL_TEXTURE_2D, 0, m_has_alpha ? GL_RGBA : GL_RGB, m_width, m_height, 0, m_has_alpha ? GL_RGBA : GL_RGB,
               GL_UNSIGNED_BYTE, image->pixels);
#endif
    
    // clean up
    SDL_UnlockSurface(image);
    SDL_FreeSurface(image);
    
    // make prettier
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
}

}
