/** @file libtrack/StartingPosition.cpp
 *  @brief Implement the Track::StartingPosition class.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "StartingPosition.h"
#include "FormErrors.h"
#include "Mesh/DrawableMesh.h"
#include "Texture.h"

#include <GL/gl.h>

namespace Track
{

std::string StartingPosition::get_class_identifier() const
{
    return std::string("Track::StartingPosition");
}

const unsigned int latest_starting_position_version = 1;

StartingPosition::StartingPosition(std::istream & source)
    :   TrackAttachment(source)
{
    unsigned int version;
    source >> version;
    if (version == 0) throw DepreciatedVersionError();
    if (version > latest_starting_position_version) throw NewVersionError();
    
    source >> m_rank;
}

StartingPosition::StartingPosition(unsigned int rank)
    :   m_rank(rank)
{
}

StartingPosition::StartingPosition()
    :   m_rank(0)
{
}

StartingPosition::~StartingPosition()
{
}

void StartingPosition::draw() const
{
    static DrawableMesh mesh("data/generic/start-marker");
    static Texture texture("data/generic/start-marker.png");
    texture.bind();
    mesh.draw();
}

void StartingPosition::add_data(std::ostream & stream) const
{
    TrackAttachment::add_data(stream);
    stream << ' ' << (*this);
}

unsigned int StartingPosition::get_rank() const
{
    return m_rank;
}

std::ostream & operator<<(std::ostream & destination, StartingPosition starting_position)
{
    destination << latest_starting_position_version << ' '
                << starting_position.get_rank();
    return destination;
}

} // namespace Track
