/** @file libtrack/SegmentConnection.h
 *  @brief Declare the Track::SegmentConnection class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_SEGMENTCONNECTION_H_
#define LIBTRACK_SEGMENTCONNECTION_H_

#include <istream>

#include <LinearMath/btTransform.h>

namespace Track
{

/** A description of an end of a TrackMesh that should be covered over by
 * another track mesh.
 * There is two basic components, an unsigned integer which identifies the cross
 * section, and a transformation. We can pick stuff with the same cross section
 * and transform them so the cross sections line up and there are no holes.
 */ 
class SegmentConnection
{
public:
    /** Construct from normal variables.
     * @param cross_section_id A name for the shape of the cross section around
     * the hole that is used throught the theme.
     * @param transform A transform from the TrackMeshes' origin to the cross
     * section's origin, so that the cross section lies across the xz plane.
     */
    SegmentConnection(unsigned int cross_section_id, btTransform transform);
    
    /** Construct by reading data from a stream.
     * @param source The stream to read from. It will be advanced to the end of
     * the SegmentConnection's description.
     */
    SegmentConnection(std::istream & source);
    
    /// destructor
    virtual ~SegmentConnection();
    
    /** Get the cross section id.
     * @return id that is used in the theme to identify the shape of the hole
     * that needs to be matched to fill the holes in TrackMeshes.
     */
    unsigned int get_cross_section_id() const;
    
    /** Get the transformation of the cross section from the origin of
     * @return transformation from the origin to the xz plane of the cross
     * section of the hole.
     */ 
     const btTransform & get_transform() const;
protected:
    unsigned int cross_section_id;
    btTransform transform;  
};

}

#endif /*LIBTRACK_SEGMENTCONNECTION_H_*/
