/** @file libtrack/Segment.h
 *  @brief Declare the Track::Segment class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_SEGMENT_H_
#define LIBTRACK_SEGMENT_H_

#include <istream>
#include <string>
#include <vector>

#include <boost/shared_ptr.hpp>

#include <LinearMath/btScalar.h>

#include "TrackMesh.h"
#include "Mesh/DrawableMesh.h"
#include "SegmentConnection.h"

namespace Track
{

/** A small, distortable, part of the track with a mesh for
 * graphics, AI navigation, the walls for the physics engine and the floors for
 * the physics engine.
 * 
 * A Segment knows how it can be connected with other segments to form
 * the track. The mesh is cut off into 'tiles' with any number of planar ends
 * to be connected to matching planar ends.
 */
class Segment
{
public:
    Segment(std::istream & source);
    virtual ~Segment();
    
    /** Get the human-readable name.
     * This name is taken from the object names in Blender when the theme is
     * exported, and displayed to the users of racer_editor.
     */
    const std::string & get_name() const;
    
    /** Get the number of joins that are used to connect these segments to
     * others. */
    std::size_t get_number_of_connections() const;
    
    /** Get segment connection information.
     * A segment defines cross sections at the entry and exit points that must
     * be connected to something with a matching cross section to form a valid
     * track layout. This function returns the information one cross section per
     * call. 
     * @param index The index from 0 to one less than the number returned by
     * get_number_of_connections() of the connectivity information to store.
     * @return Information required to determine connectivity for the requested
     * index.
     */
    const SegmentConnection & get_connection(std::size_t index) const;
    
    /** Get where the connection would be if the face was distoreted.
     * Similar to get_connection.
     * @param piece_distortion The distortion to apply to the connection information.
     * @param index The 0-based index of the connection to return the information for.
     * @return Connectivity information for the distorted piece.
     */
    const SegmentConnection & get_connection(std::size_t index,
                                                PieceDistortion piece_distortion) const;
    
    /** Get the TrackMesh used for the graphics.
     */
    const TrackMesh<MultiDrawableMesh> & get_graphics_mesh() const;
    
    /** Get the TrackMesh used for the ai's floor navigation.
     */
    const TrackMesh<> & get_ai_mesh() const;
    
    /** Get the TrackMesh used for the floor's physics.
     * This is the part you can drive on as seen by the physics engine.
     */
    const TrackMesh<> & get_floor_mesh() const;
    
    /** Get the TrackMesh used for the wall's physics.
     *  This is the part you can collide with, but not drive on, as seen by
     * the physics engine.
     */
    const TrackMesh<> & get_wall_mesh() const;
    
    /// Get the length of the segment's physics floor mesh in the y direction.
    btScalar get_length() const;
    
    /// Get the lowest y position of the vertices in the physics floor mesh.
    btScalar get_minimum_y() const;
    
    /// Return true if the mesh could be used along an edge.
    bool edges_allowed() const;
protected:
    TrackMesh<MultiDrawableMesh> graphics_mesh;
    TrackMesh<> ai_floor_mesh;
    TrackMesh<> physics_wall_mesh;
    TrackMesh<> physics_floor_mesh;
    
    /// The human-readable name. Taken from blender scene the segment came from. 
    std::string name;
    
    /// (undistorted) cross section connections information
    std::vector<boost::shared_ptr<SegmentConnection> > connections;
    
    /// True if good for edges, false otherwise.
    bool m_edges_allowed;
};

}

#endif /*LIBTRACK_SEGMENT_H_*/
