/** @file libtrack/Segment.cpp
 *  @brief Implement the Track::Segment class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "Segment.h"
#include "stream_loader.h"

namespace Track
{

Segment::Segment(std::istream & source)
    // load the meshes
    : graphics_mesh(source)
    , ai_floor_mesh(source)
    , physics_wall_mesh(source)
    , physics_floor_mesh(source)
    , name(string_from_stream(source))
{
    DEBUG_MESSAGE("Track::Segment::Segment(std::istream &) body beginning");
    // load SegmenetConnections
    fill_vector_from_stream(connections, source);
    DEBUG_MESSAGE("Loaded Track::Segment");
    
    // Decide if the segment will behave acceptably when used for edges.
    m_edges_allowed = false;
    if (connections.size() == 2)
    {
        if (connections[0]->get_cross_section_id() == connections[1]->get_cross_section_id())
        {
            // If the length of the mesh is near the distance between the
            // connections, the segment should be ok on an edge.
            // Otherwise it is probably a corner piece.
            btScalar distance2 = connections[0]->get_transform().getOrigin().
                distance2(connections[1]->get_transform().getOrigin());
            btScalar length2 = ai_floor_mesh.get_length();
            length2 *= length2;
            if (distance2 > length2 - 0.01 && distance2 < length2 + 0.01)
            {
                m_edges_allowed = true;
            }
        }
    }
}

Segment::~Segment()
{
}

const std::string & Segment::get_name() const
{
    return name;
}

std::size_t Segment::get_number_of_connections() const
{
    return connections.size();
}

const SegmentConnection & Segment::get_connection(std::size_t index) const
{
    return *(connections[index]);
}

const TrackMesh<MultiDrawableMesh> & Segment::get_graphics_mesh() const
{
    return graphics_mesh;
}

const TrackMesh<> & Segment::get_ai_mesh() const
{
    return ai_floor_mesh;
}

const TrackMesh<> & Segment::get_floor_mesh() const
{
    return physics_floor_mesh;
}

const TrackMesh<> & Segment::get_wall_mesh() const
{
    return physics_wall_mesh;
}

btScalar Segment::get_length() const
{
    return physics_floor_mesh.get_length();
}

btScalar Segment::get_minimum_y() const
{
    return physics_floor_mesh.get_minimum_y();
}

bool Segment::edges_allowed() const
{
    return m_edges_allowed;
}

}
