/** @file OcclusionTester.h
 *  @brief Declare the Track::OcclusionTester class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_OCCLUSION_TESTER_H
#define LIBTRACK_OCCLUSION_TESTER_H

#include <LinearMath/btScalar.h>

#include "AxisAlignedBoundingBox.h"

namespace Track
{

/** Tests if an object can be seen.
 * It knows the camera angle it is being tested against.
 * The camera's view frustum is passed to this class as 5 planes: we ignore
 * the back plane.
 * Objects may use OcclusionTester to test if a bounding volume is in view.
 */
class OcclusionTester
{
public:
    /// Value returned by an 
    enum ViewState
    {
        VS_OUT = 0, ///< completely outside of view area (don't draw)
        VS_IN = 1, ///< completey inside view area (draw, including all sub volumes)
        VS_PARTIAL = 2, ///<partially inside view area (draw, but you could test sub volumes individually.)
    };
    OcclusionTester();
    virtual ~OcclusionTester();
    /** Plane normal vectors.
     * A point p is in view if p dot plane_vector[i] < plane_distance[i] for all planes i.
     * Plane 0 should be the front plane.
     */
    btVector3 plane_vectors[5];
    /// The position of the camera
    btVector3 camera_position;
    /** Plane distances from the origin.
     */
    btScalar plane_distances[5];
    /// Test if an axis aligned bounding box is in view.
    ViewState operator ()(const AxisAlignedBoundingBox & aabb) const;
protected:
    /// test if a point is in a half space defined by a plane equation.
    inline bool in_half_space(const btVector3 & point, const unsigned plane_index) const;
};

} // namespace Track

#endif /*LIBTRACK_OCCLUSION_TESTER_H*/
