/** @file libtrack/NearTrack.h
 *  @brief Declare the Track::NearTrack class.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_NEAR_TRACK_H
#define LIBTRACK_NEAR_TRACK_H

#include "Mesh/MeshFaces.h"

#include <set>

namespace Track
{

/** The index of the properties returned by NearTrack::scan_direction().
 */
enum SensorIndex
{
    SENSOR_NAVIGABLE_DISTANCE = 0, ///< distance available despite curves
    SENSOR_UNSIGNED_ANGLE = 1, ///< sum of angles divided by navigable distance
    SENSOR_LAP_DISTANCE = 2, ///< the advance in lap position divided by navigable distance.
    SENSOR_LAP_DIFFERENTIAL = 3, ///< the gradient of the lap position
    SENSOR_COUNT = 4 ///< the number of sensor values calculated by NearTrack::scan_direction().
};

class Track;

/** An object which has an attachment point to an AI mesh.
 */
class NearTrack
{
    public:
        /** Construct with an initial position.
         * @param graph The navigation grap to constrain too. Cannot be
         * changed or deleted until the NearTrack is deleted.
         * @param position The nearest world space coordinates to the desired
         * starting position.
         */
        NearTrack(const MeshFaces::Graph & graph, btVector3 position);
        
        virtual ~NearTrack();
        
        /** Move the attach point closer to a position.
         * Requires an up to date AI mesh in the track.
         * 
         * The path taken will be continuous. It won't move to the nearest
         * point if the nearest point is not on a reasonable path from the
         * current location.
         * 
         * @param position The world coordinate to move towards.
         * @return square of the distance between the provided position and
         * the found point.
         */
        btScalar move_towards(btVector3 position);
        
        /** Set the attach point to the closest point on the AI navigation mesh.
         * This is much slower than move_towards(btVector3). Use that if you
         * can help it.
         * 
         * Time complexity is O(n), where n is the number of triangles in the
         * ai navigation mesh.
         * 
         * @param position. The world coordinate to jump to.
         * @return square of the distance between the provided position and
         * the found point.
         */
        btScalar jump_to(btVector3 position);
        
        /** Return the position in world space.
         */
        btVector3 get_position() const;
        
        /** Return the face the position is on.
         */
        MeshFaces::Graph::vertex_descriptor get_face_descriptor() const;
        
        /// The types of objects Track::NearTrack can attach to.
        enum AttachSource
        {
            AS_VERTEX, ///< Attachted to a Track::PathVertex
            AS_EDGE, ///< Attached to a Track::PathEdge
            AS_UNKOWN ///< Not correctly attached to anything.
        };
        
        /** Return the type of object the point is attached to.
         */
        AttachSource get_object_type() const;
        
        /** Get the identifier for the object the point is attached to.
         * Useful when combined with get_source_type.
         */
        unsigned long int get_object_name() const;
        
        /** Calculate statistics about potential movement in a given direction.
         * Projections the direction of the given vector onto the current
         * face, then finds the furthest edge in that direction. If the edge
         * is shared, project the face vector onto the next face recursively.
         * When the edge is not shared, the total navigable distance in that
         * direction is returned.
         * @param glob global vector giving the direction to scan. The length
         * of the vector is the maximum distance to scan.
         * @param results address of the first of 4 btScalars to store the
         * calculated statistics:
         * \li The inverse of the navigable distance  (1.0 / distance covered
         * by recursive  projections starting in the requested direction up to
         * the requested length),
         * \li unsigned total angle (the total change in angle, but each
         * change in angle is positive rather than depending on direction)
         * \li the total scaled lap distance (ranging from 1 when the full
         * distance is available and it is the shortest path around the lap,
         * to -1 for the direct opposite), and
         * \li the immediate differential lap distance (the rate of change in
         * lap distance on the current face in the requested direction).
         * 
         * Each result is indexed by the SensorIndex enum.
         */
        void scan_direction(btVector3 glob, btScalar * results) const;
    protected:
        /** Relative coordinates for edge or vertex attached to.
         * For an edge, this is distorted to lie along the edge.
         * The y coordinate for an edge is between 0 (start) and 1 (finish) of
         * the edge. The x and z coordinates are used like the edge's meshes.
         * 
         * For a vertex, it is in the vertex's local space.
         */
        btVector3 m_object_coord;
        
        /** The type of object the Track::NearTrack is attached to.
         * May Unkown if the relavent object is deleted from the track.
         */
        AttachSource m_object_type;
        
        unsigned long int m_object_name;
        
        /** World coordinates.
         * May become outdated when the track changes.
         */
        btVector3 m_world_coord;
        
        /** AI mesh face attached to.
         * Will be invalid when the track changes.
         */
        MeshFaces::Graph::vertex_descriptor m_face;
        
        /** Coordinates in the direction of m_face's fv1 -> fv2 edge.
         * Will be invalid when the track changes.
         */
        btScalar m_face_s;
        
        /** Coordinates in the direction of m_face's fv1 -> fv3 edge.
         * Will be invalid when the track changes.
         */
        btScalar m_face_t;
        
        /// The mesh we can navigate over.
        const MeshFaces::Graph & m_graph;
        
        /// The vertices examined in move_towards().
        std::set<MeshFaces::Graph::vertex_descriptor> examined;
        /// The lowest distance reached in move_towards().
        btScalar best_distance;
        /// The face with the lowest distance reached in move_towards().
        MeshFaces::Graph::vertex_descriptor best_face;
        /// The world coordianates of the solution with the lowest distance reached by move_towards().
        btVector3 best_world_coords;
        
        /** Find a u texture coordinate at a point on a face.
         * The u texture coordinate is used for lap distance.
         * @param face The face to inspect.
         * @param point The point in global coordinates to inspect.
         * @return Value of u texture coordinate at point, linearly
         * interpolated across the face.
         */
        static btScalar face_u_interpolation(MeshFaces::FaceGraph face, btVector3 point);
        
        /** Set m_object_coord m_object_name m_object_type, m_face_s/t from m_face.
         * Finds relavent information from the graph face.
         */
        void set_info();
        
        void move_towards_recursive(btVector3 position);
};

} // namespace Track

#endif // LIBTRACK_NEAR_TRACK_H
