/** @file libtrack/Mesh/MultiDrawableMesh.h
 *  @brief Declare the Track::MultiDrawableMesh class. 
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_MULTIDRAWABLEMESH_H
#define LIBTRACK_MULTIDRAWABLEMESH_H

#include "DrawableMesh.h"
#include "../Texture.h"

#include <istream>

namespace Track
{

extern btScalar global_quality;

/** Group of several related meshes.
 * Replaces a single mesh with several levels of detail that represent
 * the same thing. Drawing dynamically picks a mesh with reasonable
 * detail.
 * 
 * Add meshes with the add_mesh() function.
 */
class MultiDrawableMesh : public AABBDrawable
{
public:
    
    /** Construct an empty group of meshes.
     * @param render_mode The method of displaying the mesh. You can set this
     * later with set_render_mode() instead.
     */
    MultiDrawableMesh(DrawableMesh::RenderMode render_mode = DrawableMesh::RM_SOLID);
    
    /** Construct from data in a stream.
     * Moves the stream on to the end of the MultiDrawableMesh.
     */
    MultiDrawableMesh(std::istream & stream);
    
    virtual ~MultiDrawableMesh();
    
    /** Render the mesh using OpenGL commands.
     * Expects the texture and matrices to be set up already.
     * Must be called outside of a glBegin/glEnd pair.
     * Finishes with no change to the matrices, outside of a glBegin/glEnd pair.
     * Precondition: At least one mesh must have been added.
     */
    virtual void draw() const;
    
    /** Load the meshes into OpenGL to speed up their first draw.
     * Requires the OpenGL context that will be used to draw active.
     * Only needs to be called once, before draw is called.
     * Calling it after draw is called or multiple times has no additional
     * effect.
     */
    virtual void make_cache() const;
    
    /** Control how the mesh is displayed.
     * If the new RenderMode is the same as the previous, there is no effect.
     * Otherwise this will unset the cache created by a previous call to 
     * draw() or make_cache(), if any.
     * The deafult RenderMode is DrawableMesh::RenderMode::RM_SOLID.
     * @param new_render_mode The method used by draw() or make_cache().
     */
    void set_render_mode(DrawableMesh::RenderMode new_render_mode);
    
    /** Draw a suitable mesh for the view frustrum.
     * Precondition: At least one mesh must have been added.
     */
    virtual void conditional_draw(const OcclusionTester & occlusion_tester) const;
    
    /** Draw an appropriate mesh considering the distance from the camera.
     * Precondition: At least one mesh must have been added.
     * @param distance2 The square of the distance between the mesh's
     * centre and the camera.
     */
    void draw(btScalar distance2) const;
    
    /** Return the axis aligned bounding box of the first (most detailed) mesh.
     * Precondition: At least one mesh must have been added.
     */
    virtual AxisAlignedBoundingBox get_bounds() const;
    
    /** Add a mesh to the set.
     * Call this with all the meshes, in order of most detailed to least
     * detailed.
     * @param mesh Mesh to add. It will be copied.
     * @param texture Shared pointer to the texture to bind before
     * drawing the mesh.
     */
    void add_mesh(const DrawableMesh & mesh, boost::shared_ptr<Texture> texture);
    
    /** Get a mesh that has been previously added.
     * @param index the index of the mesh to return. The mesh added
     * first has index 0. The index must be less than the number
     * returned by get_num_levels().
     */
    const DrawableMesh & get_level(unsigned int index) const;
    
    /** Get a shared pointer to a texture for a mesh that has been added.
     * @param index The index of the mesh who's texture should be
     * returned. The first mesh added has index 0. index must be less
     * than the result of get_num_levels().
     */
    boost::shared_ptr<Texture> get_tex_level(unsigned int index) const;
    
    /** Return the number of mesh levels added.
     * This starts at 0, and increases by one for each call to
     * add_mesh().
     */
    unsigned int get_num_levels() const;
protected:
    /** Structure to hold a mesh and its level of detail.
     */
    struct MeshLOD
    {
        MeshLOD(const DrawableMesh & mesh, boost::shared_ptr<Texture> texture);
        DrawableMesh mesh;
        boost::shared_ptr<Texture> texture;
        // The number of triangles per cubed spacial unit.
        float triangle_density;
    };
    /// The meshes to show.
    std::vector<MeshLOD> m_meshes;
    
    AxisAlignedBoundingBox bounds;
    
    /// Method for drawing the meshes
    DrawableMesh::RenderMode m_render_mode;
    
    /// The position of the centre of the bounding box.
    btVector3 middle;
    
    /** Number approximately proportional the number of pixels covered.
     */
    btScalar area_scale;
};

} // namespace Track

#endif /*LIBTRACK_MULTIDRAWABLEMESH_H*/
