/** @file libtrack/Mesh/MultiDrawableMesh.cpp
 *  @brief Implement the Track::MultiDrawableMesh class. 
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "MultiDrawableMesh.h"
#include <Debug.h>

namespace Track
{

/// The quality of meshes selected by MultiDrawable.
/// Higher values have more triangles, and should take longer to render.
btScalar global_quality = 10.0;

MultiDrawableMesh::MultiDrawableMesh(DrawableMesh::RenderMode render_mode)
    :   m_render_mode(render_mode)
{
}

MultiDrawableMesh::MultiDrawableMesh(std::istream & stream)
    :   m_render_mode(DrawableMesh::RM_SOLID)
{
    unsigned int count;
    stream >> count;
    for (unsigned int index = 0; index < count; index++)
    {
        boost::shared_ptr<Texture> texture(new Texture(stream));
        DrawableMesh mesh(stream, m_render_mode);
        add_mesh(mesh, texture);
    }
}

MultiDrawableMesh::~MultiDrawableMesh()
{
}

void MultiDrawableMesh::draw() const
{
    // We don't know the scale.
    // Draw the highest level of detail.
    assert(!m_meshes.empty());
    const MeshLOD & lod = *m_meshes.begin();
    if (m_render_mode == DrawableMesh::RM_SOLID) lod.texture->bind();
    lod.mesh.draw();
}

void MultiDrawableMesh::make_cache() const
{
    // cache all meshes
    for (std::vector<MeshLOD>::const_iterator it = m_meshes.begin();
         it != m_meshes.end();
         it++)
    {
        it->mesh.make_cache();
        it->texture->make_cache();
    }
}

void MultiDrawableMesh::set_render_mode(DrawableMesh::RenderMode new_render_mode)
{
    for (std::vector<MeshLOD>::iterator it = m_meshes.begin();
         it != m_meshes.end();
         it++)
    {
        it->mesh.set_render_mode(new_render_mode);
    }
    m_render_mode = new_render_mode;
}

void MultiDrawableMesh::conditional_draw(const OcclusionTester & occlusion_tester) const
{
    // draw at a sensible resolution if it is on screen.
    // Get the bounds of the highest level of detail and check it is
    // on the screen first.
    assert(!m_meshes.empty());
    const DrawableMesh & highest = m_meshes.begin()->mesh;
    if (occlusion_tester(highest.get_bounds()) == OcclusionTester::VS_OUT)
    {
        // completely off screen, don't draw.
        return;
    }
    // must be at least partially on screen.
    draw(occlusion_tester.camera_position.distance2(middle));
}

void MultiDrawableMesh::draw(btScalar distance2) const
{
    assert(!m_meshes.empty());
    // Work out what sort of quality we want to use.
    // quality should be proportional area of the screen occupied, so
    // it is proportional to area_scale and inversely proportional the
    // square of the distance from the camera.
    btScalar quality = area_scale / (distance2);
    quality *= global_quality;
    // now pick a suitable mesh.
    // There are more meshes at a distance than close by, so start scanning from the back.
    for (std::vector<MeshLOD>::const_reverse_iterator it = m_meshes.rbegin();
         it != m_meshes.rend();
         it++)
    {
        // detailed enough?
        if (it->triangle_density > quality)
        {
            // this is good enough.
            if (m_render_mode == DrawableMesh::RM_SOLID)
            {
                it->texture->bind();
            }
            it->mesh.draw();
            return;
        }
    }
    // We could go more detailed than the highest level, but there isn't
    // such a mesh available. Just use the highest detailed mesh.
    if (m_render_mode == DrawableMesh::RM_SOLID)
    {
        m_meshes.begin()->texture->bind();
        m_meshes.begin()->mesh.draw();
    }
}

AxisAlignedBoundingBox MultiDrawableMesh::get_bounds() const
{
    assert(!m_meshes.empty());
    return bounds;
}

void MultiDrawableMesh::add_mesh(const DrawableMesh & mesh, boost::shared_ptr<Texture> texture)
{
    m_meshes.push_back(MeshLOD(mesh, texture));
    bounds |= mesh.get_bounds();
    if (m_meshes.size() == 1)
    {
        // first mesh
        // find some properties that will be useful later.
        AxisAlignedBoundingBox aabb = mesh.get_bounds();
        middle = (aabb.get_min() + aabb.get_max()) / 2.0;
        // area scale determines the importance of the mesh.
        //area_scale = aabb.get_volume();
        // Use the surface area at the highest detail level.
        area_scale = 0;
        for (unsigned int f_i = 0; f_i < mesh.get_number_of_faces(); f_i++)
        {
            const MeshFaces::Face & face = mesh.get_face(f_i);
            btVector3 q = mesh.get_vertex_pos(face.fv1.vertex_index);
            btVector3 r = mesh.get_vertex_pos(face.fv2.vertex_index);
            btVector3 s = mesh.get_vertex_pos(face.fv3.vertex_index);
            // technically the area is half of this, but we just want it
            // to be to a uniform scale so it doesn't matter.
            area_scale += ((q-r).cross(s-r)).length();
        }
    }
    if (m_meshes.size() > 1)
    {
        // check they are in a sensible order
        int last_faces = m_meshes.begin()->mesh.get_number_of_faces() + 1;
        for (std::vector<MeshLOD>::iterator it = m_meshes.begin();
             it != m_meshes.end();
             it++)
        {
            int this_faces = it->mesh.get_number_of_faces();
            if (last_faces <= this_faces)
            {
                std::cerr << "MultiDrawableMesh has levels of detail in a weird order:\n";
                std::cerr << "A mesh with " << this_faces << " triangles is supposidly lower detail than a mesh with " << last_faces << " triangles.\n";
            }
            last_faces = this_faces;
        }
    }
}

const DrawableMesh & MultiDrawableMesh::get_level(unsigned int index) const
{
    assert(index < m_meshes.size());
    return m_meshes[index].mesh;
}

boost::shared_ptr<Texture> MultiDrawableMesh::get_tex_level(unsigned int index) const
{
    assert(index < m_meshes.size());
    return m_meshes[index].texture;
}

unsigned int MultiDrawableMesh::get_num_levels() const
{
    return m_meshes.size();
}

MultiDrawableMesh::MeshLOD::MeshLOD(const DrawableMesh & mesh, boost::shared_ptr<Texture> texture)
    :   mesh(mesh)
    ,   texture(texture)
    ,   triangle_density(float(mesh.get_number_of_faces()) / mesh.get_bounds().get_volume())
{
    //DEBUG_MESSAGE("triangle_density is " << triangle_density);
}

} // namespace Track.
