/** @file libtrack/Mesh/MeshFaces.h
 *  @brief Declare the Track::MeshFaces class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_MESHFACES_H_
#define LIBTRACK_MESHFACES_H_

#include <vector>
#include <set>
#include <iostream>

#include <LinearMath/btVector3.h>
#include <LinearMath/btMatrix3x3.h>
#include <BulletCollision/CollisionShapes/btTriangleMesh.h>

/* gcc >= 4.3 Depreciates hash_set & hash_map, causing warnings to be emited when
 * including boost graph library headers.
 * We don't need the hash-based storage selectors anyway, so turn them off.
 */
#define BOOST_NO_HASH
#include <boost/graph/adjacency_list.hpp>
#undef BOOST_NO_HASH

#include "../AxisAlignedBoundingBox.h"
#include "../Geometry.h"

namespace Track
{

/** Describe some faces from a mesh.
 * Each face is a triangle. Stores shared vertex positions.
 * This allows us to produce a connectivity graph of faces by shared edges.
 * Use merge_doubles() to remove duplicate vertex positions, changing the
 * faces to share common vertices.
 * 
 * Vertices must be added before the faces that use them.
 */
class MeshFaces
    :   virtual public AABBBounded
{
public:
    /** Initalise from a stream.
     * The stream is advanced to the end of the MeshFaces description.
     */
    MeshFaces(std::istream & source);
    
    /** Initalise from a file.
     * @param filename The name of the file to get the mesh information from.
     */
    MeshFaces(std::string & filename);
    
    /** Construct an empty MeshFaces.
     */
    MeshFaces();
    virtual ~MeshFaces();
    
    /// vertex information for a  vertex in a face
    struct FaceVertex
    {
        /// Construct leaving all fields undefined.
        FaceVertex();
        /** Initialise just the vertex_index, leaving other fields are undefined.
         * @param vertex_index The index in the vertex_positions vector to use.
         */
        FaceVertex(std::size_t vertex_index);
        /// The index of the vertex position in the vertex positions array
        std::size_t vertex_index;
        /// The vertex normal for smooth shading
        btVector3 normal;
        /// The texture coordinate
        float texture_coord_u;
        float texture_coord_v;
    };
    
    /// A structure to store information about a single triangular face.
    struct Face
    {
        /// create leaving all fields undefined.
        Face();
        /// Create from vertex indices. No normals or texture coords set.
        Face(std::size_t v1, std::size_t v2, std::size_t v3);
        /// Create from three vertex descriptions
        Face (const FaceVertex & fv1, const FaceVertex & fv2,
              const FaceVertex & fv3);
        /// Information about the first vertex of the triangle.
        FaceVertex fv1;
        /// Information about the second vertex of the triangle.
        FaceVertex fv2;
        /// Information about the third vertex of the triangle.
        FaceVertex fv3;
        
        /// True if the face comes from an edge, false if from a vertex.
        bool is_edge;
        /// The name of the PathEdge or PathVertex the face originated from.
        unsigned long int object_name;
    };
    
    // extra information about a face useful in the graph
    struct FaceGraph : public Face
    {
        FaceGraph() {};
        FaceGraph(const Face & face);
        /// World coordinates of the average position of the vertices.
        btVector3 face_centre;
        /// World space normal of the face.
        btVector3 face_normal;
        btVector3 vertex_positions[3];
        /// World space vector from fv1 to fv2 and from fv1 to fv3
        btVector3 edges[2];
        
        /// Matrix to convert world coordinates into face coordinates
        btMatrix3x3 world_to_face;
        /// Matrix to convert face coordinates into world coordinates.
        btMatrix3x3 face_to_world;
        
        /// Plane including one of the edges, and the normal.
        Plane edge_planes[3];
        /// Edge segment for each edge.
        Line edge_line_segments[3];
        /// Plane containing the entire triangle.
        Plane face_plane;
        
        /// Set conversion matrixes, Lines and Planes from face_normal, edges, and vertex_positions.
        void set_conversions();
        
        /// convert a world space vector to face coordinates.
        btVector3 to_face_coords(btVector3 world_coords) const;
        /// convert a face coordinate vector to world space.
        btVector3 to_world_coords(btVector3 face_coords) const;
        /// Find the nearest point on the face to a world space vector in world space.
        btVector3 find_nearest_point(btVector3 position) const;
    };
    
    /// An edge between two vertices in the same face.
    struct Edge
    {
        Edge() {};
        /// Initalise from the vertex indices 
        Edge(std::size_t v1, std::size_t v2);
        /// the first vertex index to use
        std::size_t v1;
        /// the second vertex index to use
        std::size_t v2;
        /** comparison so we can order edges in a std::set.
         * Switching the order of v1 & v2 in an Edge makes no difference to its
         * ordering.
         * @param o the edge to test against
         * @return true if this edge has either vertex lower than an equivalent
         * one in o; false otherwise.
         */
        bool operator< (Edge o) const;
    };
    
    /// Extra edge information useful for the graph. 
    struct EdgeGraph : public Edge
    {
        EdgeGraph()
        {};
        
        EdgeGraph(std::size_t v1, std::size_t v2)
            :   Edge(v1, v2)
        {};
        
        btScalar length;
        btVector3 v2_pos;
        btVector3 v1_pos;
    };
    
    /// The type of graph we use to explain the navigation of the surface of the mesh.
    typedef boost::adjacency_list<boost::vecS, boost::vecS, boost::undirectedS,
                                 FaceGraph, EdgeGraph> Graph;
    
    /// Structure used for an edge in a graph showing vertices connected by face edges.
    struct FaceEdgePointers
    {
        FaceEdgePointers() {};
        FaceEdgePointers(Face * face, FaceVertex * source,
                         FaceVertex * target, btScalar length)
            :   face(face)
            ,   source(source)
            ,   target(target)
            ,   length(length)
        {};
        Face * face;
        FaceVertex * source;
        FaceVertex * target;
        btScalar length;
    };
    
    /// Type of graph that maps vertices and edges directly.
    typedef boost::adjacency_list<boost::vecS, boost::vecS, boost::undirectedS,
                                 btVector3, FaceEdgePointers> VertexGraph;

public:
    /** get the position of a vertex
     * Vertices are indexed from 0.
     */
    btVector3 get_vertex_pos(std::size_t index) const;
    
    /// add a vertex to the mesh
    void add_vertex(const btVector3 & position);
    
    /// add a vertex normal that can optionally be refered to when creating a face.
    void add_normal(const btVector3 & normal);
    
    /// add a texture coordinate that can optionally be refered to when creating a face.
    void add_texture_coord(const btVector3 & normal);
    
    /// get the number of vertices in the mesh.
    std::size_t get_num_vertices() const;
    
    /// get the vertex indices of a face
    const Face & get_face(std::size_t index) const;
    
    /// get the edges shared by a face
    std::vector<Edge> get_edges_for_face(std::size_t face) const;
    
    /** Add a face from the index of 3 vertices.
     */
    void add_face(std::size_t v1, std::size_t v2, std::size_t v3);
    
    /** Add a face from the index of 3 vertices and normals.
     */
    void add_face(std::size_t v1, std::size_t n1, std::size_t v2, std::size_t n2, std::size_t v3, std::size_t n3);
    
    /** add a face from a face description
     */
    void add_face(Face face);
    
    /// return the number of faces in the mesh
    std::size_t get_number_of_faces() const;
    
    /** Get an edge connectivity graph for all the faces.
     * Note that the vertices represent faces in the mesh.
     * The edges in the graph represent shared edges in the mesh.
     * This is because you will drive over the edges to get form one face to
     * another in most cases. The size of a vertex is 0, and our meshes are
     * well built, so we can neglect the posibility of driving across a vertex
     * to another face without crossing an edge.
     * 
     * Meshes should have the following characteristics for graph generation:
     * * An edge can only be shared by at most two faces
     * * At most one edge can be shared by any two faces.
     * * If two faces share a vertex, there must be a path across edges of faces
     * between them.
     */
    Graph get_connectivity() const;
    
    /** Make a graph of vertices and their edges.
     * Vertices in the graph represent vertices in the mesh.
     * The edges in the graph are all edges of faces.
     */
    VertexGraph get_vertex_graph();
     
     /** Copy the faces from another MeshFaces.
      * The current vertex's positions are kept, but everything else is copied
      * from the other MeshFaces. This is used to distort the Mesh by altering
      * the positions of the vertices.
      * @param other The MeshFaces to copy infromation from.
      */
    void copy_faces(const MeshFaces & other);
    
    /// Add faces in the mesh to a btTriangleMesh.
    void add_faces(btTriangleMesh & shape) const;
    
    /// Return the axis aligned bounding box of the vertices in the mesh.
    virtual AxisAlignedBoundingBox get_bounds() const;
    
    /** Merge vertices very close together.
     * Makes the connectivity graph show faces that touch, even if they didn't
     * originally share vertices. Deletes some vertices near to other ones.
     * @param distance2 The square of the distance between two vertices that
     * are close enough to be merged.
     */
    void merge_doubles(btScalar distance2 = 1.0 / 32.0);
    
    /** Add vertices and faces from another MeshFaces.
     * @param source The MeshFaces to copy data from.
     */
    void operator|=(const MeshFaces & source);
    
    /** Set the source data in all faces of the mesh.
     * Sets is_face and object_name to the same values for all faces.
     * @param is_edge true if an edge, false if a vertex.
     * @param object_name the unique object identifier of the source.
     */
    void set_source(bool is_edge, unsigned long int object_name);
protected:
    void load_from_stream(std::istream & source);
    
    void add_edge(std::size_t v1, std::size_t v2, std::size_t face_index,
                  std::map<EdgeGraph, std::size_t> & edge_to_face,
                  std::vector<std::pair<std::size_t, std::size_t> > & graph_edges,
                  std::vector<EdgeGraph> & edge_properties) const;
    
    /** Positions of vertcies.
     */
    std::vector<btVector3> vertices_position;
    
    /// Faces
    std::vector<Face> faces;
    
    AxisAlignedBoundingBox bounds;
};

}

#endif /*LIBTRACK_MESHFACES_H_*/
