/** @file libtrack/Mesh/DrawableMesh.h
 *  @brief Declare the Track::DrawableMesh class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef DRAWABLEMESH_H_
#define DRAWABLEMESH_H_

#include <boost/shared_ptr.hpp>

#include "../AABBDrawable.h"
#include "MeshFaces.h"

namespace Track
{

/** A mesh that can be drawn using OpenGL commands.
 * The mesh's faces are defined using the vertex position, normal, and texture
 * coordinates. Shaders / texturing / materials etc is left to the user.
 */
class DrawableMesh : public AABBDrawable, public MeshFaces
{
public:
    /// Method for rendering
    enum RenderMode {RM_WIREFRAME /**each face is drawn as a loop of lines*/,
                     RM_SOLID /**Complete textured faces*/};
    /** construct by sucking information from an input stream.
     * Equivalent to Track::MeshFaces::MeshFaces(std::istream &).
     * Reads data from the stream to initlalise the mesh, and leaves the stream
     * read position at the end of the DrawableMesh.
     * @param source The stream to use to initalise the mesh.
     * @param render_mode The method of displaying the mesh. You can set this
     * later with set_render_mode() instead.
     */
    DrawableMesh(std::istream & source, RenderMode render_mode = RM_SOLID);
    
    /** Construct from the contents of a file.
     * Reads data from the file specified to initlalise the mesh.
     * @param filename The filename of the file to use.
     * @param render_mode The method of displaying the mesh. You can set this
     * later with set_render_mode() instead.
     */
    DrawableMesh(std::string filename, RenderMode render_mode = RM_SOLID);
    
    /** construct from a MeshFaces obect
     * @param mesh MeshFaces to use.
     * @param render_mode The method of displaying the mesh. You can set this
     * later with set_render_mode() instead.
     */
    DrawableMesh(const MeshFaces & mesh, RenderMode render_mode = RM_SOLID);
    
    /** construct an empty mesh
     * @param render_mode The method of displaying the mesh. You can set this
     * later with set_render_mode() instead.
     */
    DrawableMesh(RenderMode render_mode = RM_SOLID);
    
    virtual ~DrawableMesh();
    
    /** Render the mesh using OpenGL commands.
     * Expects the texture and matrices to be set up already.
     * Must be called outside of a glBegin/glEnd pair.
     * Finishes with no change to the matrices, outside of a glBegin/glEnd pair.
     */
    virtual void draw() const;
    
    /** Load the mesh into OpenGL to speed up first draw.
     * Requires the OpenGL context that will be used to draw active.
     * Only needs to be called once, before draw is called.
     * Calling it after draw is called or multiple times has no additional
     * effect.
     */
    virtual void make_cache() const;
    
    /** Control how the mesh is displayed.
     * If the new RenderMode is the same as the previous, there is no effect.
     * Otherwise this will unset the cache created by a previous call to 
     * draw() or make_cache(), if any.
     * The deafult RenderMode is RM_SOLID.
     * @param new_render_mode The method used by draw() or make_cache().
     */
    void set_render_mode(RenderMode new_render_mode);
protected:
    /// OpenGL display list name where drawing instructions are stored.
    mutable unsigned int display_list;
    
    /** we use a shared_ptr so we can delete the display list when the last
     * copy of the object is deleted.
     */
    mutable boost::shared_ptr<unsigned int> list_handle;
    
    /** Delete the display list when all copies deleted.
     * It is static to meet requirements of boost::shared_ptr.
     * @param display_list the display list number to remove. Should be that of
     * the object that is being deleted.
     */
     
    static void delete_list(unsigned int * display_list);
    /** Aquire a new display list when created without copying.
     * It generates the display from the MeshFaces data.
     * @return the display_list of the object.
     */
    unsigned int * make_list() const;
    
    /// Method for drawing the mesh
    RenderMode render_mode;
};

}

#endif /*DRAWABLEMESH_H_*/
