/** @file libtrack/Mesh/DrawableMesh.cpp
 *  @brief Implement the Track::DrawableMesh class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "DrawableMesh.h"

#ifndef HAVE_GLES
#include <GL/glew.h>
#include <GL/gl.h>
#else
#include <GLES/gl.h>
#endif

#include <Debug.h>

#ifndef HAVE_GLES
// Remove to force vertex arrays instead of vertex buffer objects.
#define USE_VERTEX_BUFFER_OBJECTS
// If USE_VERTEX_BUFFER_OBJECTS is defined, VBOs are used only when supported.
// When VBOs are not supported vertex arrays are used anyway.
#endif

namespace Track
{

DrawableMesh::DrawableMesh(std::istream & source, RenderMode render_mode)
    :   MeshFaces(source)
    ,   display_list(0)
    ,   render_mode(render_mode)
{
}

DrawableMesh::DrawableMesh(const MeshFaces & mesh, RenderMode render_mode)
    :   MeshFaces(mesh)
    ,   display_list(0)
    ,   render_mode(render_mode)
{
}

DrawableMesh::DrawableMesh(std::string filename, RenderMode render_mode)
    :   MeshFaces(filename)
    ,   display_list(0)
    ,   render_mode(render_mode)
{
}

DrawableMesh::DrawableMesh(RenderMode render_mode)
    :   display_list(0)
    ,   render_mode(render_mode)
{
}

DrawableMesh::~DrawableMesh()
{
}

void DrawableMesh::draw() const
{
    if (!display_list)
    {
        // we haven't made the display list for the mesh yet. Do that now.
        make_cache();
    }
#ifndef HAVE_GLES
    glCallList(display_list);
}

unsigned int * DrawableMesh::make_list() const
{
    // OpenGL vertex buffer names for mesh data.
#endif
    unsigned int buffer_name;
    std::vector<GLfloat> data;
    // normal, tex u, tex v, position; for 3 vertices a face
    const unsigned int elements_per_vertex = 8;
    data.reserve(faces.size() * 3 * (elements_per_vertex));
    for (std::vector<MeshFaces::Face>::const_iterator it = faces.begin();
         it != faces.end();
         it++)
    {
#define do_vert(v)\
        data.push_back(v.normal.getX());\
        data.push_back(v.normal.getY());\
        data.push_back(v.normal.getZ());\
        data.push_back(v.texture_coord_u);\
        data.push_back(v.texture_coord_v);\
        data.push_back(vertices_position[v.vertex_index].getX());\
        data.push_back(vertices_position[v.vertex_index].getY());\
        data.push_back(vertices_position[v.vertex_index].getZ());
        
        do_vert(it->fv1)
        do_vert(it->fv2)
        do_vert(it->fv3)
#undef do_vert
    }
    assert(data.size() == faces.size() * 3 * elements_per_vertex);
    float * pointer_offset = 0;
#ifdef USE_VERTEX_BUFFER_OBJECTS
    if (GLEW_ARB_vertex_buffer_object)
    {
        // use a vertex buffer object if supported.
        glGenBuffersARB(1, &buffer_name);
        glBindBufferARB(GL_ARRAY_BUFFER_ARB, buffer_name);
        glBufferDataARB(GL_ARRAY_BUFFER_ARB,
                        sizeof(GLfloat) * data.size(),
                        &(data[0]),
                        GL_STATIC_DRAW_ARB);
    } else {
#endif
        // Use vertex arrays.
        pointer_offset = &(data[0]);

#ifdef USE_VERTEX_BUFFER_OBJECTS
    }
#endif
    glEnableClientState(GL_NORMAL_ARRAY);
    glNormalPointer(GL_FLOAT, elements_per_vertex * sizeof(GLfloat), pointer_offset);

    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glTexCoordPointer(2, GL_FLOAT, elements_per_vertex * sizeof(GLfloat), pointer_offset + 3);
    
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, elements_per_vertex * sizeof(GLfloat), pointer_offset + 5);
        
#ifndef HAVE_GLES
    display_list = glGenLists(1);
    glNewList(display_list, GL_COMPILE);
#endif
        switch (render_mode)
        {
            case RM_SOLID:
                glDrawArrays(GL_TRIANGLES, 0, faces.size() * 3);
                break;
            case RM_WIREFRAME:
                for (unsigned int i = 0; i < faces.size(); i++)
                {
                    glDrawArrays(GL_LINE_LOOP, i * 3, 3);
                }
                break;
        }
#ifndef HAVE_GLES
    glEndList();
#endif
    
    glDisableClientState(GL_NORMAL_ARRAY);
    glDisableClientState(GL_TEXTURE_COORD_ARRAY);
    glDisableClientState(GL_VERTEX_ARRAY);
    
#ifdef USE_VERTEX_BUFFER_OBJECTS
    if (GLEW_ARB_vertex_buffer_object)
    {
        glDeleteBuffersARB(1, &buffer_name);
    }
#endif

#ifdef HAVE_GLES
}

unsigned int * DrawableMesh::make_list() const
{
	unsigned int display_list = 0;
#endif
    return &display_list;
}

void DrawableMesh::delete_list(unsigned int * display_list)
{
#ifndef HAVE_GLES
    glDeleteLists(*display_list, 1);
#endif
}

void DrawableMesh::make_cache() const
{
    if (!display_list)
    {
        list_handle = boost::shared_ptr<unsigned int>(make_list(), delete_list);
    }
}

void DrawableMesh::set_render_mode(RenderMode new_render_mode)
{
    if (new_render_mode == render_mode) return;
    render_mode = new_render_mode;
    // this cleans up the display list if there was one already.
    list_handle = boost::shared_ptr<unsigned int>();
}

}
