/** @file libtrack/Mesh/BulletMesh.h
 *  @brief Declare the Track::BulletMesh class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_BULLET_MESH_H
#define LIBTRACK_BULLET_MESH_H

#include <string>
#include <vector>
#include <utility>

#include <btBulletDynamicsCommon.h>
#include <btBulletCollisionCommon.h>

#include "DrawableMesh.h"

namespace Track
{

/** A DrawableMesh which can produce shapes for the Bullet physics library.
 * When created you should use bit masks for the type of shape you want to
 * produce:
 * a convex hull (fastest for moving objects),
 * a triangle mesh for the bullet physics library (good for static
 * geometry), or both.
 * 
 */
class BulletMesh
    :   public DrawableMesh
{   
    // no public copy constructor, since we could delete Bullet objects in use. 
    BulletMesh(const BulletMesh & source);
    // no assignment operator, since we could delete Bullet objects in use.
    BulletMesh & operator=(const BulletMesh & source);
public:
    /// bit mask to create a convex hull of the points in the mesh for bullet.
    static const unsigned int genererator_convex_hull_bit = 2;
    /// bit mask to create a triangle mesh of the mesh for bullet.
    static const unsigned int genererator_triangle_mesh_bit = 4;
    
    /** Construct some objects from the mesh specifed by the file given.
     * @param filename The name and path of a file containing the mesh to
     * load.
     * @param generators A bit mask made from the above three constants to
     * determine which objects to create.
     * @param render_mode The method to use to draw the mesh.
     */  
    BulletMesh(std::string filename,
                 unsigned int generators = genererator_triangle_mesh_bit,
                 DrawableMesh::RenderMode render_mode = DrawableMesh::RM_SOLID);
    
    /** Construct some objects from the mesh specified by a stream.
     * @param generators A bit mask made from the generator constants to
     * determine which objects to create.
     * @param source The stream to load the mesh from.
     * @param render_mode The method to use to draw the mesh.
     */
    BulletMesh(std::istream & source,
                 unsigned int generators = genererator_triangle_mesh_bit,
                 DrawableMesh::RenderMode render_mode = DrawableMesh::RM_SOLID);
    
    /// A type used to construct the mesh. Works exactly like Mesh(std::string, unsigned int).
    typedef std::pair<std::string, unsigned int> ConstructionInformation;
    
    /// Construct from a ConstructionInformation object.
    BulletMesh(ConstructionInformation construction_information);
    
    virtual ~BulletMesh();
    
    /** Get the convex hull of the vertices in the mesh.
     * It should be efficent for calculating physics when used for a moving
     * object.
     * @return Pointer to convex hull shape, which deleted with the BulletMesh.
     */
    btConvexHullShape * get_convex_hull_shape();
    /** Get a Bullet btBvhTriangleMeshShape representing the faces in the mesh.
     * It is uesful for static geometry.
     * @return Pointer to triangle mesh shape, which deleted with the BulletMesh.
     */
    btBvhTriangleMeshShape * get_triangle_shape();
private:
    /// Create the triangle / convex hull shape
    void create(unsigned int generators = genererator_triangle_mesh_bit);
    
    // pointers to the shapes created.
    // 0 if the apprpriate bit was not set when creating the mesh.
    btConvexHullShape * simplified_convex_shape;
    btBvhTriangleMeshShape * triangle_mesh_shape;
    btTriangleMesh * triangle_mesh;
};

}

#endif /*LIBTRACK_BULLET_MESH_H*/
