/** @file libtrack/Mesh/BulletMesh.cpp
 *  @brief Implement the Track::BulletMesh class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "BulletMesh.h"

#include <fstream>

#include <GL/gl.h>
#include <btBulletDynamicsCommon.h>
#include <btBulletCollisionCommon.h>
#include <BulletCollision/CollisionShapes/btShapeHull.h>

#include <Debug.h>

namespace Track
{

BulletMesh::BulletMesh(std::string filename, unsigned int generators, DrawableMesh::RenderMode mode)
    :   DrawableMesh(filename, mode)
    ,   simplified_convex_shape(0)
    ,   triangle_mesh_shape(0)
    ,   triangle_mesh(0)
{
    create(generators);
}

BulletMesh::BulletMesh(std::istream & source, unsigned int generators, DrawableMesh::RenderMode mode)
    :   DrawableMesh(source, mode)
    ,   simplified_convex_shape(0)
    ,   triangle_mesh_shape(0)
    ,   triangle_mesh(0)
{
    create(generators);
}

BulletMesh::BulletMesh(ConstructionInformation construction_information)
    :   DrawableMesh(construction_information.first.c_str())
    ,   simplified_convex_shape(0)
    ,   triangle_mesh_shape(0)
    ,   triangle_mesh(0)
{
    create(construction_information.second);
}

void BulletMesh::create(unsigned int generators)
{ 
    bool make_convex = generators & genererator_convex_hull_bit;
    bool make_triangle = generators & genererator_triangle_mesh_bit;
    
    btConvexHullShape base_shape;
    if (make_triangle)
    {
        triangle_mesh = new btTriangleMesh();
    }
    {
        if (make_convex)
        {
            for (std::vector<btVector3>::const_iterator it = vertices_position.begin();
                 it != vertices_position.end();
                 it++)
            {
                base_shape.addPoint(*it);
            }
        }
        if (make_triangle)
        {
            for (std::vector<MeshFaces::Face>::const_iterator it = faces.begin();
                 it != faces.end();
                 it++)
            {
                triangle_mesh->addTriangle(vertices_position[it->fv1.vertex_index],
                                           vertices_position[it->fv2.vertex_index],
                                           vertices_position[it->fv3.vertex_index]);
            }
        }
    }

    // simplify the shape with a hull approximation
    if (make_convex)
    {
        btShapeHull hull(&base_shape);
        btScalar margin = base_shape.getMargin();
        hull.buildHull(margin);
        simplified_convex_shape = new btConvexHullShape(&(hull.getVertexPointer()->m_floats[0]),
                                                        hull.numVertices());
    }
    
    // make the triangle mesh shape
    if (make_triangle)
    {
        triangle_mesh_shape = new btBvhTriangleMeshShape(triangle_mesh, false);
    }
}

BulletMesh::~BulletMesh()
{
    DEBUG_MESSAGE("Deleting mesh");
    delete simplified_convex_shape; 
    delete triangle_mesh_shape;
    delete triangle_mesh;
}

btConvexHullShape * BulletMesh::get_convex_hull_shape()
{
    return simplified_convex_shape;
}

btBvhTriangleMeshShape * BulletMesh::get_triangle_shape()
{
    return triangle_mesh_shape;
}

} // namespace Track
