/** @file libtrack/Lighting.h
 *  @brief Declare the Track::Lighting, Track::LightSettings, and Track::FogSettings classes.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_LIGHTING_H_
#define LIBTRACK_LIGHTING_H_

#include <istream>
#include <ostream>
#include <vector>

namespace Track
{

/** A light that illuminates the track and cars on it.
 * Configure these through Track::Lighting.
 */
struct LightSettings
{
    LightSettings();
    LightSettings(std::istream & in);
    /** Global xyzw position of light.
     * w=0 sets the light infinetly far away.
     */
    float position[4];
    /// RGBA colour of the light
    float colour[4];
    bool operator == (const LightSettings & other) const;
};

/// Serialize a Track::LightSettings.
std::ostream & operator<<(std::ostream & destination, const LightSettings & light_settings);

/** A graphical effect that blends distant objects to a colour.
 * Configure through Track::Lighting.
 */
struct FogSettings
{
    FogSettings();
    FogSettings(std::istream & in);
    /// RGBA colour of the fog
    float colour[4];
    /** Fog thickness.
     * Must be > 0.
     * The fog is blended with a factor of e^(-(density*distance)^2).
     */
    float density;
    /// true iff the fog is used at all.
    bool enabled;
    bool operator != (const FogSettings & other) const;
};

/// Serialize a Track::FogSettings.
std::ostream & operator<<(std::ostream & destination, const FogSettings & fog_settings);

/** Defines lighting and atmospheric effects for a track.
 * @todo Support positional lighting. Currently only directional
 * lighting will work (where the light's position[3] is 0).
 */
class Lighting
{
    public:
        // Initalise to sensible defaults.
        Lighting();
        
        /// Create from a serialised record.
        Lighting(std::istream & in);
        
        virtual ~Lighting();
        
        /** Set up lighting and fog for a render.
         * Should be called for each render, between setting the matrices and
         * drawing geometry.
         */
        void prepare_render() const;
        
        /** Initalise lighting and fog in OpenGL.
         * Should be called before rendering any frame.
         */
        void initalise() const;
        
        /** Set the lights' settings.
         */
        void set_lights(const std::vector<LightSettings> & lights);
        /** Get the lights' settings.
         */
        const std::vector<LightSettings> & get_lights() const;
        
        /** Set the ambient light colour.
         */
        void set_ambient(float colour[4]);
        /** Get the ambient light colour.
         */
        const float * get_ambient_light() const;
        
        /** Set the fog settings.
         */
        void set_fog(const FogSettings & fog);
        /** Get the fog settings
         */
        const FogSettings & get_fog() const;
        
        bool operator!=(const Lighting & other) const;
        
        /** Turn off lighting and fog.
         * For drawing things like the sky and HUD.
         */
        void turn_off() const;
        /** Turn on the lighting and fog.
         * Use prepare_render as well if the camera moved.
         */
        void turn_on() const;
    private:
        std::vector<LightSettings> m_lights;
        FogSettings m_fog;
        float m_ambient_light[4];
        /** OpenGL shader for drawing track with this lighting.
         * @todo this is mutable because it doesn't affect the lighting,
         * but the shader needs to be cached. However, it is required
         * for turn_on() and turn_off() to work as expected, so maybe
         * turn_on() should create the shader if needed instead of
         * initialise()?
         */
        mutable unsigned int m_shader_program_handle;
        mutable unsigned int m_vertex_shader_handle;
        mutable unsigned int m_fragment_shader_handle;
        
        /** Free shaders and the shader program (if previously created).
         * If they were not created (i.e. remain at the inital value 0),
         * no action is taken.
         */
        void free_shaders() const;
};

std::ostream & operator<<(std::ostream & destination, const Lighting & lighting);

} // Track namespace

#endif // LIBTRACK_LIGHTING_H_
