/** @file libtrack/Lighting.cpp
 *  @brief Implement the Track::Lighting, Track::LightSettings, and Track::FogSettings classes.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "Lighting.h"

#ifndef HAVE_GLES
#include <GL/glew.h>
#include <GL/gl.h>
#else
#include <GLES/gl.h>
#endif
#include "stream_loader.h"
#include "FormErrors.h"

#include <sstream>
#include <cmath>
#include <cstring>

#include <LinearMath/btTransform.h>


namespace Track
{

const unsigned int lighting_newest_version = 1;
const unsigned int fog_settings_newest_version = 1;
const unsigned int light_settings_newest_version = 1;

FogSettings::FogSettings()
    :   density(0.004)
    ,   enabled(true)
{
    const float default_fog_colour[4] = {0.3, 0.0, 0.1, 1.0};
    for (int i = 0; i < 4; i++)
    {
        colour[i] = default_fog_colour[i];
    }
}

FogSettings::FogSettings(std::istream & in)
{
    unsigned int file_version;
    in >> file_version;
    if (file_version == 0) throw CorruptedError();
    if (file_version > fog_settings_newest_version) throw NewVersionError();
    in >> density >>
          colour[0] >> colour[1] >> colour[2] >> colour[3] >>
          enabled;
}

bool FogSettings::operator != (const FogSettings & other) const
{
    if (enabled != other.enabled) return true;
    if (density != other.density) return true;
    for (int i = 0; i < 4; i++)
    {
        if (colour[i] != other.colour[i]) return true;
    }
    return false;
}

std::ostream & operator<<(std::ostream & dest, const FogSettings & fog)
{
    dest << fog_settings_newest_version << ' ' <<
            fog.density << ' ' <<
            fog.colour[0] << ' ' << fog.colour[1] << ' ' <<
            fog.colour[2] << ' ' << fog.colour[3] << ' ' <<
            fog.enabled;
    return dest;
}

LightSettings::LightSettings()
{
}

LightSettings::LightSettings(std::istream & in)
{
    unsigned int file_version;
    in >> file_version;
    if (file_version == 0) throw CorruptedError();
    if (file_version > light_settings_newest_version) throw NewVersionError();
    in >> position[0] >> position[1] >> position[2] >> position[3] >>
          colour[0] >> colour[1] >> colour[2] >> colour[3];
}

bool LightSettings::operator == (const LightSettings & other) const
{
    for (int i = 0; i < 4; i++)
    {
        if (position[i] != other.position[i]) return false;
        if (colour[i] != other.colour[i]) return false;
    }
    return true;
}

std::ostream & operator<<(std::ostream & dest, const LightSettings & ls)
{
    dest << light_settings_newest_version << ' ' <<
            ls.position[0] << ' ' << ls.position[1] << ' ' <<
            ls.position[2] << ' ' << ls.position[3] << ' ' <<
            ls.colour[0]   << ' ' << ls.colour[1]   << ' ' <<
            ls.colour[2]   << ' ' << ls.colour[3];
    return dest;
}

Lighting::Lighting()
    :   m_shader_program_handle(0)
    ,   m_vertex_shader_handle(0)
    ,   m_fragment_shader_handle(0)
{
    m_lights.resize(2);
    const float l0p[4] = {0.0, 0.0, 1.0, 0.0};
    const float l0c[4] = {0.4, 0.4, 0.4, 1.0};
    const float l1p[4] = {1.0, 0.0, 0.0, 0.0};
    const float l1c[4] = {0.4, 0.2, 0.1, 1.0};
    const float amb[4] = {0.6, 0.6, 0.6, 1.0};
    for (int i = 0; i < 4; i++)
    {
        m_lights[0].position[i] = l0p[i];
        m_lights[0].colour[i] = l0c[i];
        m_lights[1].position[i] = l1p[i];
        m_lights[1].colour[i] = l1c[i];
        m_ambient_light[i] = amb[i];
    }
}

Lighting::Lighting(std::istream & in)
    :   m_shader_program_handle(0)
    ,   m_vertex_shader_handle(0)
    ,   m_fragment_shader_handle(0)
{
    unsigned int file_version;
    in >> file_version;
    if (file_version == 0) throw CorruptedError();
    if (file_version > lighting_newest_version) throw NewVersionError();
    m_fog = FogSettings(in);
    in >> m_ambient_light[0] >> m_ambient_light[1] >>
          m_ambient_light[2] >> m_ambient_light[3];
    unsigned int m_num_lights;
    in >> m_num_lights;
    if (m_num_lights > 8) throw CorruptedError();
    m_lights.resize(m_num_lights);
    for (int i = 0; i < m_num_lights; i++)
    {
        m_lights[i]= LightSettings(in);
    }
}

Lighting::~Lighting()
{
    free_shaders();
}

void Lighting::free_shaders() const
{
#ifndef HAVE_GLES
    // If there is no OpenGL context, glDelete* could crash.
    // Lighting can be destroyed without an OpenGL context in the editor
    // because loading a track replaces the default Lighting with
    // one found in a file. It can't do it in the initaliser list
    // because old files don't have Lighting and require the default
    // constructor instead of the stream constructor.
    // So, even  though the documentation states glDeleteShader(0) has
    // no effect, we check for 0 first.
    if (m_shader_program_handle)
    {
        // The lighting is only destroyed when the track is, so shaders
        // are not needed. Go back to fixed function.
        glUseProgram(0);
        // delete the shader
        glDeleteProgram(m_shader_program_handle);
    }
    if (m_vertex_shader_handle)
    {
        glDeleteShader(m_vertex_shader_handle);
    }
    if (m_fragment_shader_handle)
    {
        glDeleteShader(m_fragment_shader_handle);
    }
#endif
}

/** Print the shader or program's compile log to standard output.
 * @param obj The handle for the object who's log will be displayed.
 * @param program True for a program, false for a vertex or fragment shader.
 */
void printShaderInfoLog(GLuint obj, bool program = false)
{
#ifndef HAVE_GLES
	int info_log_length = 0;
    if (program)
    {
        glGetProgramiv(obj, GL_INFO_LOG_LENGTH, &info_log_length);
    } else {
        glGetShaderiv(obj, GL_INFO_LOG_LENGTH, &info_log_length);
    }

	if (info_log_length > 0)
	{
        char * info_log = (char *)malloc(info_log_length);
        if (info_log)
        {
            int charsWritten;
            if (program)
            {
                glGetProgramInfoLog(obj, info_log_length, &charsWritten,
                                    info_log);
            } else {
                glGetShaderInfoLog(obj, info_log_length, &charsWritten,
                                   info_log);
            }
            std::cout << info_log << std::endl;
            free(info_log);
        } else {
            std::cerr << "Error allocating buffer for shader's info log.\n";
        }
	}
#endif
}

void Lighting::initalise() const
{
    // lights
    for (unsigned int i = 0; i < m_lights.size(); i++)
    {
        glEnable(GL_LIGHT0 + i);
        glLightfv(GL_LIGHT0 + i, GL_DIFFUSE, m_lights[i].colour);
    }
    // disable unused lights that could have activated from a previous game.
    for (unsigned int i = m_lights.size(); i < 8; i++)
    {
        glDisable(GL_LIGHT0+i);
    }
    // ambient light
    glLightModelfv(GL_LIGHT_MODEL_AMBIENT, m_ambient_light);
    // only reflects a fith of the ambient light and 4 fiths of diffuse light
    // by default. Use all light.
    GLfloat light_scale[4] = {1.0, 1.0, 1.0, 1.0};
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, light_scale);
    
    // fog
    glFogf(GL_FOG_MODE, GL_EXP2);
    glFogf(GL_FOG_DENSITY, m_fog.density);
    glFogfv(GL_FOG_COLOR, m_fog.colour);
    if (m_fog.enabled) glEnable(GL_FOG); else
                     glDisable(GL_FOG);
    
    // Make an OpenGL shader if it is supported.
#ifndef HAVE_GLES
    if (GLEW_ARB_vertex_shader && GLEW_ARB_fragment_shader)
    {
        // Free shaders created last time, if any.
        free_shaders();
        // Vertex shader
        std::stringstream vertex_source;
        vertex_source << std::fixed
            << "varying vec4 light;" << std::endl;
        if (m_fog.enabled)
        {
            vertex_source << "varying float fog_density;" << std::endl;
        }
        vertex_source
            << "varying vec2 tex_coords;" << std::endl
            
            << "void main() {" << std::endl
            << "    gl_Position = ftransform();" << std::endl
            << "    vec3 N = (gl_NormalMatrix * gl_Normal).xyz;" << std::endl
            << "    tex_coords = gl_MultiTexCoord0.xy;" << std::endl;
        if (m_fog.enabled)
        {
            vertex_source
                // The lens is not wide angle enough to justify the
                // square root required for calculating the exact
                // distance, which is length(gl_Position).
                << "    float eye_distance = gl_Position.z;" << std::endl
                // The fog should be calcualted like we do in fixed
                // function pipeline, with GL_FOG_MODE set to GL_EXP2.
                << "    fog_density = "
                        << "exp(eye_distance * eye_distance * "
                        << -(m_fog.density * m_fog.density)
                        << ");" << std::endl;
        }
        vertex_source
            << "    vec4 light_total = vec4(" <<
                               m_ambient_light[0] << ", " <<
                               m_ambient_light[1] << ", " <<
                               m_ambient_light[2] << ", " <<
                               m_ambient_light[3] << ");"  << std::endl
            << "    vec3 L;" << std::endl
            << "    float diffuse, specular;" << std::endl;
        // add contribution for each light
        for (unsigned int light_index = 0; light_index < m_lights.size(); light_index++)
        {
            vertex_source
                << "    L= gl_LightSource[" << light_index << "].position.xyz;" << std::endl
                << "    diffuse = max(dot(N, L), 0.0);" << std::endl
                << "    specular = pow(max(dot(N, gl_LightSource[" << light_index << "].halfVector.xyz), 0.0), 20.0) * 0.5;" << std::endl
                << "    light_total += (diffuse + specular) * " <<
                // light colour
                        "vec4(" << m_lights[light_index].colour[0] << ", "
                        << m_lights[light_index].colour[1] << ", "
                        << m_lights[light_index].colour[2] << ", "
                        << m_lights[light_index].colour[3] << ");" << std::endl;
        }
        vertex_source
            // the multipler brightens the visible edges in diffuse shader.
            << "    float multiplier = 1.25 - (N.z * N.z) * 0.5;" << std::endl
// I think this way looks better, but isn't worth the processing cost.
//            << "    vec3 V = normalize((gl_ModelViewMatrix * gl_Vertex).xyz);" << std::endl
//            << "    float multiplier = 1 + dot(N, V) * 0.7;" << std::endl
            // gamma adjustment to reduce underexposure and overexposure.
            << "    light = pow(light_total * multiplier, vec4(0.45, 0.45, 0.45, 1));" << std::endl
            << "}" << std::endl;
        char * vertex_shader_cstr = new char[vertex_source.str().size() + 1];
        std::strcpy(vertex_shader_cstr, vertex_source.str().c_str());
        m_vertex_shader_handle = glCreateShader(GL_VERTEX_SHADER);
        glShaderSource(m_vertex_shader_handle, 1, const_cast<const char**>(&vertex_shader_cstr), 0);
        glCompileShader(m_vertex_shader_handle);
#ifndef NDEBUG
        std::cout << "Vertex shader source:\n";
        std::cout << vertex_shader_cstr;
        std::cout << "Vertex shader compile log:\n";
        printShaderInfoLog(m_vertex_shader_handle);
#endif
        delete[] vertex_shader_cstr;
        
        // Fragment shader
        std::stringstream fragment_source;
        fragment_source
            <<  "varying vec4 light;" << std::endl
            <<  "varying vec2 tex_coords;" << std::endl;
        if (m_fog.enabled)
        {
            fragment_source << "varying float fog_density;" << std::endl;
        }
        fragment_source << std::fixed
            << "uniform sampler2D base_sampler;" << std::endl

            << "void main() {" << std::endl
            << "    vec4 base_tex = texture2D(base_sampler, tex_coords);" << std::endl
            << "    gl_FragColor = light * base_tex;" << std::endl;
        if (m_fog.enabled)
        {
            // mix in the fog colour
            fragment_source
                <<  "    gl_FragColor = mix(" <<
                //fog colour
                "vec4(" << m_fog.colour[0] << ", "
                        << m_fog.colour[1] << ", "
                        << m_fog.colour[2] << ", "
                        << m_fog.colour[3] << ")"
                << ", gl_FragColor, fog_density);" << std::endl;
        }
        // close main function
        fragment_source << "}" << std::endl;
        char * fragment_shader_cstr = new char[fragment_source.str().size() + 1];
        std::strcpy(fragment_shader_cstr, fragment_source.str().c_str());
        m_fragment_shader_handle =  glCreateShader(GL_FRAGMENT_SHADER);
        glShaderSource(m_fragment_shader_handle, 1, const_cast<const char**>(&fragment_shader_cstr), 0);
        
        glCompileShader(m_fragment_shader_handle);
#ifndef NDEBUG
        std::cout << "Fragment shader source:\n";
        std::cout << fragment_shader_cstr;
        std::cout << "Fragment shader compile log:\n";
        printShaderInfoLog(m_fragment_shader_handle);
#endif
        delete[] fragment_shader_cstr;
        
        // link the shader program.
        m_shader_program_handle = glCreateProgram();
        glAttachShader(m_shader_program_handle, m_fragment_shader_handle);
        glAttachShader(m_shader_program_handle, m_vertex_shader_handle);
        glLinkProgram(m_shader_program_handle);
#ifndef NDEBUG
        std::cout << "Shader program link log:\n";
        printShaderInfoLog(m_shader_program_handle, true);
#endif
        
        glUseProgram(m_shader_program_handle);
        /** @todo stop leaking Shader objects.
         */
    }
#endif
}

void Lighting::prepare_render() const
{
    glEnable(GL_LIGHTING);
    if (m_fog.enabled) glEnable(GL_FOG); else
                     glDisable(GL_FOG);
    for (unsigned int i = 0; i < m_lights.size(); i++)
    {
        // normalize light direction
        float scale = 0;
        for (int c = 0; c < 3; c++)
        {
            scale += m_lights[i].position[c] * m_lights[i].position[c];
        }
        scale = std::sqrt(scale);
        float l[4];
        for (int c = 0; c < 3; c++)
        {
            l[c] = m_lights[i].position[c] / scale;
        }
        l[3] = 0.0;
        // tell OpenGL.
        glLightfv(GL_LIGHT0 + i, GL_POSITION, l);
    }
}

void Lighting::set_lights(const std::vector<LightSettings> & lights)
{
    m_lights = lights;
}

const std::vector<LightSettings> & Lighting::get_lights() const
{
    return m_lights;
}

void Lighting::set_ambient(float colour[4])
{
    for (int i = 0; i < 4; i++)
    {
        m_ambient_light[i] = colour[i];
    }
}

const float * Lighting::get_ambient_light() const
{
    return m_ambient_light;
}

void Lighting::set_fog(const FogSettings & fog)
{
    m_fog = fog;
}

void Lighting::turn_off() const
{
#ifndef HAVE_GLES
    glUseProgram(0);
#endif
    glDisable(GL_LIGHTING);
    glDisable(GL_FOG);
}

void Lighting::turn_on() const
{
#ifndef HAVE_GLES
    glUseProgram(m_shader_program_handle);
#endif
    // If shader is broken or not compatible with hardware/driver,
    // we need the fixed function fallback enabled.
    glEnable(GL_LIGHTING);
    glEnable(GL_FOG);
}

const FogSettings & Lighting::get_fog() const
{
    return m_fog;
}


bool Lighting::operator!=(const Lighting & other) const
{
    if (m_lights != other.m_lights) return true;
    if (m_fog != other.m_fog) return true;
    for (int i = 0; i < 4; i++)
    {
        if (m_ambient_light[i] != other.m_ambient_light[i]) return true;
    }
    return false;
}

std::ostream & operator<<(std::ostream & dest, const Lighting & lighting)
{
    const float * amb = lighting.get_ambient_light();
    dest << lighting_newest_version << ' ' <<
            lighting.get_fog() << ' ' <<
            amb[0] << ' ' << amb[1] << ' ' << amb[2] << ' ' << amb[3] << ' ';
    const std::vector<LightSettings> & lights = lighting.get_lights();
    dest << lights.size();
    for (int i = 0; i < lights.size(); i++)
    {
        dest << ' ' << lights[i];
    }
    return dest;
}

} // namespace Track.
