/** @file Libtrack/Geometry.h
 *  @brief Declare the Track::Plane, and Track::Line classes. 
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_GEOMETRY_H
#define LIBTRACK_GEOMETRY_H

#include <LinearMath/btVector3.h>
#include <cmath>

namespace Track
{

/// A three dimensional plane.
class Plane
{
    public:
        /** Make a plane containg points p such that vector * p = distance.
         *
         * If used as a half space, find_side will return true if the point
         * lies further in the direction of vector. Negate distance and vector
         * to define the same plane but the opposite half space.
         */
        Plane(btVector3 vector, btScalar distance)
            :   m_vector(vector)
            ,   m_distance(distance)
        {
            normalise();
        }
        /// Construct without initalising plane.
        Plane()
        {
        };
        /// Determine which side of the plane a point is on
        inline bool find_side(btVector3 point) const
        {
            return find_signed_distance(point) < 0.0;
        }
        /// Find minimum distance to the plane.
        inline btScalar find_distance(btVector3 point) const
        {
            return std::abs(find_signed_distance(point));
        }
        /// Find signed minimum distance to the plane.
        inline btScalar find_signed_distance(btVector3 point) const
        {
            return point.dot(m_vector) + m_distance;
        }
        /// Find the nearest point to the plane.
        inline btVector3 find_nearest_point(btVector3 point) const
        {
            return point - m_vector * find_signed_distance(point);

        }
    protected:
        btVector3 m_vector;
        btScalar m_distance;
        /** Put plane in Hessian normal form.
         * This makes m_vector unit length, but keeps the object describing
         * the same plane. It means that some functions are simpler as they
         * do not need to consider the length of m_vector.
         */
        inline void normalise()
        {
            btScalar length = m_vector.length();
            m_vector = m_vector / length;
            m_distance /= length;
        }
};

/// A three dimensional line, half line, or line segment.
class Line
{
    public:
        Line(btVector3 begin, btVector3 end)
            :   m_begin(begin)
            ,   m_end(end)
            ,   m_vector(end - begin)
            ,   m_length2(m_vector.length2())
        {
        }
        Line()
        {}
        /** Find the closest point on the line segment to the given point.
         * @param point The point to check.
         * @return value indicating distance along line. 0 is the beginning,
         * and 1 is the end of the line segment.
         */
        inline btScalar closest_parametric(btVector3 point) const
        {
            return (point - m_begin).dot(m_vector) / m_length2;
        }
        /** Like closest_parametric but clamped to fit between the begining
         * and the end point.
         */
        inline btScalar closest_parametric_segment(btVector3 point) const
        {
            btScalar d = closest_parametric(point);
            if (d < 0.0) return 0.0;
            if (d > 1.0) return 1.0;
            return d;
        }
        
        inline btVector3 closest_point_segment(btVector3 point) const
        {
            return m_begin + m_vector * closest_parametric_segment(point);
        }
        /** Return a plane that contains the line and the direction of vector.
         */
        inline Plane get_plane_including(btVector3 vector) const
        {
            // Find vector for plane normal.
            btVector3 plane_normal = m_vector.cross(vector);
            plane_normal = plane_normal.normalized();
            return Plane(plane_normal, -(plane_normal.dot(m_begin)));
        }
    protected:
        btVector3 m_begin;
        btVector3 m_end;
        /// m_end - m_begin.
        btVector3 m_vector;
        /// square of the length of m_vector.
        btScalar m_length2;
};

} // namespace Track

#endif // LIBTRACK_GEOMETRY_H
