/** @file libtrack/DataCheck.cpp
 *  @brief Implement the Track::DataCheck class.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "DataCheck.h"
#include <fstream>

/** @todo Make some Windows equivalent since unistd.h's chdir is a posix thing.
 */
#include <unistd.h>

#include <Debug.h>

namespace Track
{

DataCheck::DataCheck()
{
    try
    {
        check_file("data/generic/booster");
    }
    catch (FileMissingError e)
    {
        /**@todo The installation prefix, $(prefix) in autotools, might not
         * be /usr/local. Find the path properly.
         */
        // Not found. Try using ./racer (works when in the trunk folder)
        if (!check_relative_directory("racer"))
        {
            // Try looking up a level? (works in racer_editor or libtrack)
            if (!check_relative_directory("../racer"))
            {
                // up one level? Should work in racer/data etc.
                if (!check_relative_directory(".."))
                {
                    // Not found in current directory. Has it been installed?
                    if (!check_directory("/usr/local/share/racer"))
                    {
                        // How about with a package manager?
                        if (!check_directory("/usr/share/racer"))
                        {
                            // give up finding it.
                            throw e;
                        }
                    }
                }
            }
        }
    }
}

void DataCheck::check_file(std::string filename)
{
    std::ifstream file(filename.c_str());
    if (file.fail())
    {
        FileMissingError error;
        error.filename = filename;
        throw error;
    }
}

bool DataCheck::check_directory(std::string path)
{
    try
    {
        check_file(path + "/data/generic/booster");
        // must have worked, set directory.
    }
    catch (FileMissingError e)
    {
        // not here.
        return false;
    }
    // directory works, change it to the working directory.
    if (chdir(path.c_str()) == -1)
    {
        // Unusable, even though we found the file here.
        DEBUG_MESSAGE("Cannot change working directory, but data files found in " << path << ".");
        return false;
    }
    return true;
}

bool DataCheck::check_relative_directory(std::string path_relative)
{
    char path_cs[1024];
    if (getcwd(path_cs, 1024) == 0)
    {
        DEBUG_MESSAGE("Cannot get current working directory, so cannot see if data files are somewhere relative.");
        FileMissingError e;
        e.filename = "data/generic/booster";
        throw e;
    }
    std::string path_full(path_cs);
    path_full += "/" + path_relative;
    return check_directory(path_full);
}

} // namespace Track

