/** @file libtrack/ClassLoader.h
 *  @brief Declare the Track::SaveableClassList and Track::ClassLoader classes.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_CLASS_LOADER_H
#define LIBTRACK_CLASS_LOADER_H

#include <iostream>
#include <map>
#include <vector>
#include <string>

#include <boost/shared_ptr.hpp>

#include "FormErrors.h"
#include "stream_loader.h"

namespace Track
{

/** A class that an object that can be loaded even if the class changes.
 * Any object to be loaded with SavableClassList must derive from this.
 */
class Savable
{
    /** Return the string used to identify the class.
     * This is the same string as the associated ClassLoaderBase uses.
     * The return value should be the same for each instance of the class.
     */
    virtual std::string get_class_identifier() const = 0;
};

/** A non-template class with the properties we want of a ClassLoader.
 */
class  ClassLoaderBase
{
    public:
        /// Load an object from a input stream once the type is known.
        virtual boost::shared_ptr<Savable> create(std::istream & source) const = 0;
        
        /// Get the identifier associated with the class this object can load.
        virtual std::string get_identifier() const = 0;
};

template <class T>
class ClassLoader;

/// The version of the data written about an object to find its class at load time.
const unsigned int latest_class_record_version = 1;
/// The version of a list of classes of unknown type.
const unsigned int latest_class_record_vector_version = 1;

/** Initalize the SavableClassList.
 * On first call, creates lots of ClasssLoaders. After that does nothing.
 */
void register_all_classes_with_savable_class_list();

/** Read and write objects of any class derived from Savable from/to a file.
 * This is a singleton object, you may access the single instance with
 * get_instance().
 * To register a class to be loaded in this manner, make a global ClassLoader.
 */
class SaveableClassList
{
    private:
        SaveableClassList();
        SaveableClassList(const SaveableClassList & source);
        bool operator=(const SaveableClassList & source);
    public:
        /// Get the instance of the class.
        static SaveableClassList * get_instance();
        
        /// register a class so we may load data from it.
        template <class T>
        void register_class(ClassLoader<T> * loader)
        {
            loaders.insert(std::pair<std::string, ClassLoader<T> *>(loader->get_identifier(), loader));
        }
        
        /// Load an object from a stream
        template<class T>
        boost::shared_ptr<T> load_object(std::istream & source)
        {
            register_all_classes_with_savable_class_list();
            unsigned int version;
            source >> version;
            if (version < latest_class_record_version)
            {
                throw DepreciatedVersionError();
            }
            else if (version > latest_class_record_version)
            {
                throw NewVersionError();
            }
            std::string type;
            type = string_from_stream(source);
            std::map<std::string, ClassLoaderBase *>::iterator it = loaders.find(type);
            if (it == loaders.end())
            {
                // couldn't find the object, probably a new one.
                assert(false);
                throw UnkownObjectError();
            }
            boost::shared_ptr<T> ptr = boost::dynamic_pointer_cast<T, Savable>(it->second->create(source));
            if (!ptr)
            {
                // Not a base class of the requested type.
                // Classes changed so this is invalid.
                assert(false);
                throw WrongTypeError();
            }
            return ptr;
        }
        
        /** Write an object so that it can be loaded again without knowing the type.
         * Requires the class will return the identifier it's ClassLoader uses
         * with a function std::string get_class_identifier(), and will
         * work with operator<< to streams.
         */
        template <class T>
        void write_object(std::ostream & destination, const T & object)
        {
            destination << latest_class_record_version << " ";
            string_to_stream(destination, object.get_class_identifier());
            destination << ' ' << object;
        }
        
        /** Load many objects from a stream.
         */
        template <class T>
        void fill_vector(std::istream & source, std::vector<boost::shared_ptr<T> > & vector)
        {
            unsigned int version;
            source >> version;
            if (version < latest_class_record_vector_version)
            {
                throw DepreciatedVersionError();
            }
            else if (version > latest_class_record_vector_version)
            {
                throw NewVersionError();
            }
            // how many items were saved?
            std::size_t size;
            source >> size;
            vector.reserve(size);
            // load each one.
            for (unsigned int index = 0; index < size; index++)
            {
                boost::shared_ptr<T> ptr(load_object<T>(source));
                vector.push_back(ptr);
            }
        }
        
        /** Write many objects to a stream.
         * @tparam Container should store boost::shared_ptrs to the type of
         * object you wish to save. It should follow the forward container
         * concept from the standard template library.
         */
        template <class Container>
        void write_vector(std::ostream & destination, const Container & vector)
        {
            destination << latest_class_record_vector_version << ' ';
            destination << vector.size();
            for (typename Container::const_iterator it = vector.begin();
                 it != vector.end();
                 it++)
            {
                destination << ' ';
                write_object(destination, **it);
            }
        }
    private:
        std::map<std::string, ClassLoaderBase *> loaders;
};

/** An object which will be used to load a class from a file, when we are not
 * sure of the exact type of the object when loading.
 * Must not be destroyed before a file is loaded, as it is required to decode
 * it then. It registers itself with the SavableClassInstance when created,
 * but does not unregister when destroyed.
 * @param T The class to create.
 */
template <class T>
class ClassLoader
    :   public ClassLoaderBase
{
    public:
        /** Create a ClassLoader for a specific class.
         * This must be a global variable, as it must not be deleted before
         * a class is loaded.
         * @param name the Unique name to associate with when loading the 
         * class of a file.
         * It is recommended to be the class name including namespace.
         */
        ClassLoader(std::string name)
            :   m_name(name)
        {
            SaveableClassList::get_instance()->register_class(this);
        }
        
        /** Create an object by reading its info from a file.
         * @param source The file to read from. The file will be advanced to
         * the end of the loaded object.
         * @return A boost shared pointer to the newly loaded object.
         */
        virtual boost::shared_ptr<Savable> create(std::istream & source) const
        {
            boost::shared_ptr<Savable> result(new T(source));
            return result;
        }
        
        /// Get the unique identification string for the class.
        virtual std::string get_identifier() const
        {
            return m_name;
        }
    protected:
        /// The unique identifier for the class.
        std::string m_name;
};

} // namespace Track

#endif // LIBTRACK_CLASS_LOADER_H
