/** @file libtrack/AxisAlignedBoundingBox.h
 *  @brief Declare the Track::AxisAlignedBoundingBox class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_AXIS_ALIGNED_BOUNDING_BOX_H_
#define LIBTRACK_AXIS_ALIGNED_BOUNDING_BOX_H_

#include <LinearMath/btVector3.h>
#include <LinearMath/btScalar.h>
#include <LinearMath/btTransform.h>

namespace Track
{

/** A three dimensional volume that can be expressed as a cuboid, where each
 * face is perpendicular to an axis.
 * 
 * It is generally used to find the extents of some object.
 * 
 * For example:
 * @code AxisAlignedBoundingBox box;
btVector3 vertex;
...
box |= vertex; @endcode
 * Then <tt>box</tt> is expanded to include <tt>vertex</tt>. If all vertices
 * were added this way, then <tt>box</tt> would be the bounding volume.
 * 
 * You can take a newly created bounding box and <tt>operator&=</tt> all the
 * vertices of your object to find its bounding volume.
 */
class AxisAlignedBoundingBox
{
public:
    /** Create a new bounding box. It is empty until you add vertices into it.
     */
    AxisAlignedBoundingBox();
    virtual ~AxisAlignedBoundingBox();
    
    /** Include a point in the bounding box, expanding as necessary.
     * @param point point to include in the box.
     */
    void operator|=(const btVector3 & point);
    
    /// include another axis aligned bounding box.
    void operator|=(const AxisAlignedBoundingBox & other);
    
    /** Add some padding around each side of the box.
     * @param size the width to add around the edge of the box.
     */
    void add_border(btScalar size);
    
    /** Check if the bounding box contains at least one point.
     * @return true if the bouding box contains at least one point, false if it
     *         has been created without adding any points, or a negative border
     *         has been applied that inverses the volume.
     */
    bool valid() const;
    
    /// The largest x coordinate in the volume.
    btScalar get_max_x() const;
    /// The largest y coordinate in the volume.
    btScalar get_max_y() const;
    /// The largest z coordinate in the volume.
    btScalar get_max_z() const;
    /// The minimum x coordinate in the volume.
    btScalar get_min_x() const;
    /// The minimum y coordinate in the volume.
    btScalar get_min_y() const;
    /// The minimum z coordinate in the volume.
    btScalar get_min_z() const;
    
    /// Get all the minimum coordinates in the volume.
    btVector3 get_min() const;
    
    /// Get all the maximum coordinates in the volume.
    btVector3 get_max() const;
    
    /// Get the length of the bounding box in each direction.
    btVector3 get_extent() const;
    
    /// Calculate the volume of the bounding box.
    btScalar get_volume() const;
    
    /** Find the minimum axis aligned bounding box of a transformed bounding
     * box.
     * The box may not be minimal for it's original contents, because the
     * details inside the box are lost. Chaining transformations of Axis
     * aligned bounding boxes together will make the box increasingly large.
     * It is recommended to use this only on an original object aligned
     * bounding box to transform it to world space.
     */
    AxisAlignedBoundingBox transform(btTransform transform) const;
protected:
    /// All the minimum coordinates
    btVector3 min;
    /// All the maximum coordinates.
    btVector3 max;
};

/** An object that can find its AxisAlignedBoundingBox.
 * Abstract base class.
 */
class AABBBounded
{
    public:
        /** Get the Axis aligned bounding box of the object.
         * The object is contained entierly within this region.
         */
        virtual AxisAlignedBoundingBox get_bounds() const = 0;
};

}

#endif /* LIBTRACK_AXIS_ALIGNED_BOUNDING_BOX_H_ */ 
